/******************************************************************
 *FILE: atmel_mxt_ts.c
 *SW-COMPONENT: QNX atmel touch driver
 *DESCRIPTION: This source file is for atmel touch core functionalities.
 *COPYRIGHT: © 2019 Robert Bosch GmbH
 *
 *This program is free software; you can redistribute  it and/or modify it
 *under  the terms of  the GNU General  Public License as published by the
 *Free Software Foundation;  either version 2 of the  License, or (at your
 *option) any later version.
 ******************************************************************/
#include <stddef.h>
#include <string.h>
#include <stdint.h>
#include <stdio.h>
#include <stdbool.h>
#include <errno.h>
#include <fcntl.h>
#include <time.h>
#include <unistd.h>
#include <pthread.h>
#include <malloc.h>
#include "faceplate_rm.h"
#include "fidm_touch.h"
#include "stdarg.h"
#include <limits.h>

#include "atmel_mxt_ts.h"
#include "atmel_mxt_i2c.h"

#ifndef UNITTEST
#include <devctl.h>
#include <hw/i2c.h>
#include <sys/neutrino.h>
#include <sys/slog.h>
#include <gulliver.h>
#include "dcmd_pm.h"
#include "input/mtouch_log.h"
#include "input/mtouch_driver.h"
#include "input/event_types.h"
#include "input/inputtrace.h"
#include "input/parseopts.h"
#include "utils.h"
#include "lib_display_binder.h"
#else
#include "ipc_header.h"
#include "input_utest_header.h"
#include "pthread_header.h"
#include "i2c_header.h"
#include "err_mem_header.h"
#include "slog2_header.h"
#include "interrupt_header.h"
#include "touchscreen_header.h"
#endif

#ifndef UNITTEST
#define atmel_static static
#else
#define atmel_static
#endif

#define ATTACH_POINT "atmel_chn"
#define ATMEL_CONFIG_PATH   "/etc/system/config/"

//#define min(x,y) ( (x < y) ? x : y )

uint8_t count = 0;
bool report_id_count_flag = false;
bool report_id_flag = false;
bool nonsequential_reportid_flag = false;

atmel_static int mxt_send_bootloader_cmd(struct mxt_data *data, bool unlock);
static int atmel_get_display_grpid(struct mxt_data *atmel);

void proc_first_msg(struct mxt_data *data, uint8_t *msg);
void proc_sec_to_four_msg(struct mxt_data *data, uint8_t *msg);
void proc_last_msg();
void reset_state_variable();
/* Write critical logs into errmem */
void error_memory(const char * fmt, ...)
{
#define MAXLINE    1024   /* max text line length */

    va_list ap;
    char buf[MAXLINE] = {0};

    va_start(ap, fmt);
    vsnprintf(buf, sizeof(buf), fmt, ap);
    va_end(ap);

    vWritePrintfErrmem(buf);
}

int atmel_get_display_grpid(struct mxt_data *atmel)
{
    int handle, ret;
    char *reply;
    reply = NULL;

    if(!atmel->dev_status_path)
    {
        mtouch_error(ATMEL_DEV_NAME, "Not a valid status path %s", atmel->dev_status_path);
        return -EINVAL;
    }
    handle = displaybinder_status_open(atmel->dev_status_path);
    if(handle < 0)
    {
        mtouch_error(ATMEL_DEV_NAME, "Failed to open display binder status path: %s, handle:%d", atmel->dev_status_path, handle);
        return handle;
    }
    mtouch_debug(ATMEL_DEV_NAME, "display binder open success(reval: %d) for dev status path %s", handle, atmel->dev_status_path);
    ret = displaybinder_status_get(handle,"display-group-id",&reply);
    displaybinder_status_close(handle);
    if (ret) {
       mtouch_error(ATMEL_DEV_NAME, "failed to fetch group id: %d", ret);
    } else {
        atmel->display_group_id = atoi(reply);
        mtouch_info(ATMEL_DEV_NAME, "Display Group ID is fetched successfully: %d", atmel->display_group_id);
    }

    if(reply)
       free(reply);
    return ret;
}

/**
 * Will sleep for a minum of usec even if the sleep call is interrupted.
 */
int
safe_usleep(useconds_t usec)
{
    struct timespec ts;
    int rc;

    ts.tv_sec = usec / 1000000;
    ts.tv_nsec = (usec % 1000000) * 1000;
    while (1) {
        rc = nanosleep(&ts, &ts);
        if (EINTR != rc) {
            break;
        }
    }

    return rc;
}

atmel_static int
atmel_get_contact_id(void* packet, uint8_t digit_idx, uint32_t* contact_id, void* arg)
{
    *contact_id = digit_idx;
    return 0;
}

atmel_static int
atmel_is_contact_down(void* packet, uint8_t digit_idx, int* valid, void* arg)
{
    struct mxt_touch_state *touch_state = packet;
    struct mxt_data* atmel = arg;

    *valid = !!(touch_state[digit_idx].status & MXT_T100_DETECT);
    if(atmel->verbose > 9)
        mtouch_debug(ATMEL_DEV_NAME, "Contact %d down: %s", digit_idx, *valid ? "yes" : "no");

    return 0;
}

atmel_static int
atmel_get_coords(void* packet, uint8_t digit_idx, int32_t* x, int32_t* y, void* arg)
{
    struct mxt_touch_state* touch_state = packet;
    struct mxt_data* atmel = arg;
    //struct timespec ts;

    *x = touch_state[digit_idx].x;
    *y = touch_state[digit_idx].y;

    //clock_gettime(CLOCK_MONOTONIC, &ts);
    //mtouch_debug(ATMEL_DEV_NAME, "t%d x: %d, y: %d, %d ms", digit_idx, *x, *y, ts.tv_nsec/1000000);
    if (atmel->verbose) {
        mtouch_info(ATMEL_DEV_NAME, "RAW Touch Coordinates: t%d x: %d, y: %d", digit_idx, *x, *y);
    }

    atmel->last_coords.x = *x;
    atmel->last_coords.y = *y;

    return EOK;
}

int atmel_get_touch_width(void* packet, _Uint8t digit_idx, _Uint32t* touch_width, void* arg)
{
    struct mxt_data* atmel = arg;
    struct mxt_touch_state* touch_state = packet;

    *touch_width = touch_state[digit_idx].width;

    if (atmel->verbose > 6)
        mtouch_info (ATMEL_DEV_NAME, "Touch width sent to screen: Finger %d width: %d", digit_idx, *touch_width);

    return EOK;
}

int atmel_get_touch_height(void* packet, _Uint8t digit_idx, _Uint32t* touch_height, void* arg)
{
    struct mxt_data* atmel = arg;
    struct mxt_touch_state* touch_state = packet;

    *touch_height = touch_state[digit_idx].height;

    if (atmel->verbose > 6)
        mtouch_info (ATMEL_DEV_NAME, "Touch height sent to screen: Finger %d height: %d", digit_idx, *touch_height);

    return EOK;
}

int atmel_get_touch_pressure(void* packet, _Uint8t digit_idx, _Uint32t* touch_pressure, void* arg)
{
    struct mxt_data* atmel = arg;
    struct mxt_touch_state* touch_state = packet;

    *touch_pressure = touch_state[digit_idx].pressure;

    if (atmel->verbose > 6)
        mtouch_info (ATMEL_DEV_NAME, "Touch pressure sent to screen: Finger %d pressure: %d", digit_idx, *touch_pressure);
    return EOK;
}

int atmel_get_touch_orientation(void* packet, _Uint8t digit_idx, _Uint32t* touch_orientation, void* arg)
{
    struct mxt_data* atmel = arg;
    struct mxt_touch_state* touch_state = packet;

    *touch_orientation = touch_state[digit_idx].orientation;

    if (atmel->verbose > 6)
        mtouch_info (ATMEL_DEV_NAME, "Touch orientation sent to screen: Finger %d orientation: %d", digit_idx, *touch_orientation);
    return EOK;
}

atmel_static void
get_seq_id(void* packet, uint32_t* seq_id, void* arg)
{
    struct mxt_data* atmel = arg;
    *seq_id = atmel->seq_id++;
    atmel->seq_id &= ~(INPUTTRACE_SEQ_TYPE(INPUTTRACE_SEQ_TYPE_MASK));
}

int
atmel_get_contact_type(void* packet, uint8_t digit_idx, uint32_t* contact_type, void* arg)
{
    struct mxt_touch_state* touch_state = packet;
    int type = CONTACT_TYPE_FINGER;

    *contact_type = type;  /* Default */

    /* Only T series of controller supports this currently */
    type = ((touch_state[digit_idx].status & MXT_T100_TYPE_MASK) >> 4);

    switch (type) {
    case MXT_T100_TYPE_FINGER:
        *contact_type = CONTACT_TYPE_FINGER;
        break;
    case MXT_T100_TYPE_PASSIVE_STYLUS:
        *contact_type = CONTACT_TYPE_STYLUS;
        break;
    case MXT_T100_TYPE_ACTIVE_STYLUS:
        *contact_type = CONTACT_TYPE_STYLUS;
        break;
    case MXT_T100_TYPE_HOVERING_FINGER:
        *contact_type = CONTACT_TYPE_HOVERING;
        break;
    case MXT_T100_TYPE_GLOVE:
        *contact_type = CONTACT_TYPE_GLOVE;
        break;
    case MXT_T100_TYPE_LARGE_TOUCH:
        *contact_type = CONTACT_TYPE_LARGE;
        break;
    default:
        mtouch_error (ATMEL_DEV_NAME, "Unknown contact type %d", type);
    }

    return EOK;
}

/**
 * Attach to libinputevents
 */
static int
attach_driver(struct mxt_data* dev)
{
    mtouch_driver_funcs_t funcs = {
        .get_contact_id = atmel_get_contact_id,
        .is_contact_down = atmel_is_contact_down,
        .get_coords = atmel_get_coords,
        .get_down_count = NULL,
        .get_touch_width = NULL,
        .get_touch_height = NULL,
        .get_touch_orientation = NULL,
        .get_touch_pressure = atmel_get_touch_pressure,
        .get_seq_id = get_seq_id,
        .set_event_rate = NULL,
        .get_contact_type = atmel_get_contact_type,
        .get_select = NULL
    };

    mtouch_driver_params_t params = {
        .capabilities = MTOUCH_CAPABILITIES_CONTACT_ID | MTOUCH_CAPABILITIES_COORDS | MTOUCH_CAPABILITIES_SEQ_ID | MTOUCH_CAPABILITIES_PRESSURE,
        .flags = 0,
        .max_touchpoints = MXT_MAX_NUM_TOUCHPTS,
        .width = dev->width,
        .height = dev->height
    };

    mtouch_info(ATMEL_DEV_NAME, "%s compiled on " __DATE__ " at " __TIME__, __FUNCTION__);

    dev->inputevents_hdl = mtouch_driver_attach(&params, &funcs);

    if (NULL == dev->inputevents_hdl) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to connect to libinputevents");
        error_memory("Atmel_Touch: Failed to connect to libinputevents");
        return -1;
    }

    return EOK;
}

static struct mxt_object *
mxt_get_object(struct mxt_data *data, uint8_t type)
{
    struct mxt_object *object;
    int i;

    for (i = 0; i < data->info->object_num; i++) {
        object = data->object_table + i;
        if (object->type == type)
            return object;
    }

    mtouch_warn(ATMEL_DEV_NAME, "Invalid object type T%u\n", type);
    return NULL;
}

static void mxt_free_object_table(struct mxt_data *data)
{
    data->object_table = NULL;
    free(data->info);
    data->info = NULL;
    free(data->msg_buf);
    data->msg_buf = NULL;
    data->T5_address = 0;
    data->T5_msg_size = 0;
    data->T6_reportid = 0;
    data->T7_address = 0;
    data->T71_address = 0;
    data->T9_reportid_min = 0;
    data->T9_reportid_max = 0;
    data->T15_reportid_min = 0;
    data->T15_reportid_max = 0;
    data->T18_address = 0;
    data->T19_reportid = 0;
    data->T25_address = 0;
    data->T25_reportid = 0;
    data->T42_reportid_min = 0;
    data->T42_reportid_max = 0;
    data->T44_address = 0;
    data->T48_reportid = 0;
    data->T92_reportid = 0;
    data->T92_address = 0;
    data->T93_reportid = 0;
    data->T93_address = 0;
    data->T100_reportid_min = 0;
    data->T100_reportid_max = 0;
    data->T152_reportid_min = 0;
    data->T152_reportid_max = 0;
    data->T152_address = 0;
    data->T33_reportid_min = 0;
    data->T33_reportid_max = 0;
    data->T33_address = 0;
    data->max_reportid = 0;
    memset(data->touch_prev_rec, 0, sizeof(data->touch_prev_rec)); // Initialize touch_prev_rec to 0
}

static void mxt_calc_crc24(uint32_t *crc, uint8_t firstbyte, uint8_t secondbyte)
{
    static const unsigned int crcpoly = 0x80001B;
    uint32_t result;
    uint32_t data_word;

    data_word = (secondbyte << 8) | firstbyte;
    result = ((*crc << 1) ^ data_word);

    if (result & 0x1000000)
        result ^= crcpoly;

    *crc = result;
}

static uint32_t mxt_calculate_crc(uint8_t *base, off_t start_off, off_t end_off)
{
    uint32_t crc = 0;
    uint8_t *ptr = base + start_off;
    uint8_t *last_val = base + end_off - 1;

    if (end_off < start_off)
        return -EINVAL;

    while (ptr < last_val) {
        mxt_calc_crc24(&crc, *ptr, *(ptr + 1));
        ptr += 2;
    }

    /* if len is odd, fill the last byte with 0 */
    if (ptr == last_val)
        mxt_calc_crc24(&crc, *ptr, 0);

    /* Mask to 24-bit */
    crc &= 0x00FFFFFF;

    return crc;
}

static size_t mxt_obj_size(const struct mxt_object *obj)
{
    return obj->size_minus_one + 1;
}

static size_t mxt_obj_instances(const struct mxt_object *obj)
{
    return obj->instances_minus_one + 1;
}

static int mxt_t6_command(struct mxt_data *data, uint16_t cmd_offset,
              uint8_t value, bool wait)
{
    uint16_t reg;
    uint8_t command_register;
    int timeout_counter = 0;
    int ret, rc;

    reg = data->T6_address + cmd_offset;

    ret = data->i2c_funcs.write_reg(data->i2c_fd, reg, 1, &value);
    if (ret)
        return ret;

    if (!wait)
        return 0;

    do {
        rc = safe_usleep(20 * 1000);
        if(rc != 0)
        {
            mtouch_error(ATMEL_DEV_NAME, "mxt_t6_command:Timer error");
        }
        ret = data->i2c_funcs.read_reg(data->i2c_fd, reg, 1, &command_register);
        if (ret)
            return ret;
    } while (command_register != 0 && timeout_counter++ <= 100);

    if (timeout_counter > 100) {
        mtouch_error(ATMEL_DEV_NAME, "Command failed!\n");
        error_memory("Atmel_Touch: Command failed!\n");
        return -EIO;
    }

    return 0;
}

#if 0
static int mxt_t25_command(struct mxt_data *data, uint8_t cmd, bool wait)
{
    uint16_t reg;
    int timeout_counter = 0;
    int ret, rc;
    uint8_t  val[2];

    reg = data->T25_address;
    val[0] = 0x3;
    val[1] = cmd;

    data->t25_status = true;
    ret = data->i2c_funcs.write_reg(data->i2c_fd, reg, sizeof(val), val);
    if (ret) {
        data->t25_status = false;
        return ret;
    }

    if (!wait)
        return 0;

    do {
        rc = safe_usleep(MXT_WAKEUP_TIME * 1000);
        if(rc != 0)
        {
            mtouch_error(ATMEL_DEV_NAME, "mxt_t25_command:Timer error");
        }
        ret = data->i2c_funcs.read_reg(data->i2c_fd, reg + 1, 1, &val[1]);
        if (ret)
            return ret;
    } while ((val[1] != 0) && (timeout_counter++ <= 100));

    if (timeout_counter > 100) {
        mtouch_error(ATMEL_DEV_NAME, "Command failed!\n");
        data->t25_status = false;
        return -EIO;
    }
    return 0;
}
#endif

static int mxt_soft_reset(struct mxt_data *data)
{
    int ret = 0, rc;

    mtouch_info(ATMEL_DEV_NAME, "Resetting device\n");

    InterruptMask(data->tp_intr, data->tp_iid);
    ret = mxt_t6_command(data, MXT_COMMAND_RESET, MXT_RESET_VALUE, false);
    if (ret)
        return ret;

    /* Ignore CHG line for 100ms after reset */
    rc = safe_usleep(MXT_RESET_INVALID_CHG * 1000);
    if(rc != 0)
    {
        mtouch_error(ATMEL_DEV_NAME, "mxt_soft_reset:Timer error");
    }
    InterruptUnmask(data->tp_intr, data->tp_iid);

    return 0;
}

static int mxt_read_blks(struct mxt_data *data, uint16_t start, uint16_t count, uint8_t *buf)
{
    uint16_t offset = 0;
    int error;
    uint16_t size;

    while (offset < count) {
        size = min(MXT_MAX_BLOCK_WRITE, count - offset);
        error = data->i2c_funcs.read_reg(data->i2c_fd, start + offset,
                       size, buf + offset);
        if (error < 0) {
            mtouch_error(ATMEL_DEV_NAME,"failed to read i2c device,%d", error);
            return error;
        }
        offset += size;
    }

    return 0;
}

static void mxt_dump_message(struct mxt_data *data, uint8_t *message)
{
    int i;

    if (data->verbose > 7) {
        mtouch_debug(ATMEL_DEV_NAME, "MXT MSG: %d", data->T5_msg_size);
        for (i = 0; i < data->T5_msg_size; i++)
            mtouch_debug(ATMEL_DEV_NAME, "MXT MSG: %d", message[1]);
    }
}

static void mxt_proc_t25_messages(struct mxt_data *data, uint8_t *msg)
{
    /* Output debug if status has changed */
    mtouch_debug(ATMEL_DEV_NAME, "T25 Status 0x%x Info: %x %x %x %x %x\n",
        msg[1],
        msg[2],
        msg[3],
        msg[4],
        msg[5],
        msg[6]);

    /* Save current status */
    memcpy(&data->t25_msg[0], &msg[1], sizeof(data->t25_msg));
    data->t25_status = false;
}

static void mxt_proc_t92_messages(struct mxt_data *data, uint8_t *msg)
{
    uint8_t status = msg[1];

    mtouch_info(ATMEL_DEV_NAME, "T92 long stroke LSTR=%d %d\n",
         (status & 0x80) ? 1 : 0,
         status & 0x0F);
}

static void mxt_proc_t93_messages(struct mxt_data *data, uint8_t *msg)
{
    uint8_t status = msg[1];

    mtouch_info(ATMEL_DEV_NAME, "T93 report double tap %d\n", status);
}

static void mxt_proc_t15_messages(struct mxt_data *data, uint8_t *msg)
{
#if 0
    int key;
    bool curr_state, new_state;
    bool sync = false;
//TODO work on T15 keys
    unsigned long keystates = le32_to_native((__force __le32)msg[2]);

    for (key = 0; key < data->t15_num_keys; key++) {
        curr_state = test_bit(key, &data->t15_keystatus);
        new_state = test_bit(key, &keystates);

        if (!curr_state && new_state) {
            dev_dbg(dev, "T15 key press: %u\n", key);
            __set_bit(key, &data->t15_keystatus);
            input_event(input_dev, EV_KEY,
                    data->t15_keymap[key], 1);
            sync = true;
        } else if (curr_state && !new_state) {
            dev_dbg(dev, "T15 key release: %u\n", key);
            __clear_bit(key, &data->t15_keystatus);
            input_event(input_dev, EV_KEY,
                    data->t15_keymap[key], 0);
            sync = true;
        }
    }

    if (sync)
        input_sync(input_dev);
#endif
}

static int mxt_knob_connect_faceplate(struct mxt_data *data)
{
    char *attach_point;
    attach_point = strdup(FACEPLATE_ATTACH_POINT);
    data->faceplate_coid = name_open(attach_point, 0);
    if (data->faceplate_coid == -1)
    {
        mtouch_error(ATMEL_DEV_NAME, "name_open failed for FACEPLATE_ATTACH_POINT, error: %d\n",errno);
        return -1;
    }
    mtouch_info(ATMEL_DEV_NAME, "Connected to faceplate driver successfully, coid = %d\n", data->faceplate_coid);

    return 0;
}

static void mxt_proc_t152_messages(struct mxt_data *data, uint8_t *msg)
{
    int ret;
    faceplate_msg send_msg;

    mtouch_info(ATMEL_DEV_NAME,"Received knob data: %x %x %x %x %x %x %x %x %x %x \n",
                                msg[0],
                                msg[1],
                                msg[2],
                                msg[3],
                                msg[4],
                                msg[5],
                                msg[6],
                                msg[7],
                                msg[8],
                                msg[9]);

    if (data->faceplate_coid == -1)
    {
        mtouch_info(ATMEL_DEV_NAME,"Establishing connection to faceplate\n");
        ret = mxt_knob_connect_faceplate(data);
        if(ret != 0)
        {
            mtouch_error(ATMEL_DEV_NAME, "Connection failed to faceplate retval = %d\n", ret);
            return;
        }
    }
    send_msg.message.command = FACEPLATE_KNOB_SUPPORT;
    send_msg.message.buff = (int32_t)msg[3];
    ret = MsgSend(data->faceplate_coid, &send_msg, sizeof(send_msg), NULL, 0);
    if (ret != EOK)
    {
        mtouch_error(ATMEL_DEV_NAME," MsgSend to faceplate is failed error = %d",errno);
    }
    mtouch_info(ATMEL_DEV_NAME,"knob data sent to faceplate : %d\n", send_msg.message.buff);
}

atmel_static void mxt_proc_t33_messages(struct mxt_data *data, uint8_t *msg)
{

  if(msg[0] == data->T33_reportid_min)
     reset_state_variable();

  switch(count)
  {
    case FIRST_MSG_PKT:
#ifdef BOSCH_RTC_2574261_HID_ENABLE_ATMEL_LOGGING_IMPROVEMENTS
        data->Atmel_rtd_update_recvd_counter++;
        // Reset the counter if it reaches the maximum value to prevent overflow
        if (data->Atmel_rtd_update_recvd_counter >= UINT32_MAX) {
            mtouch_warn(ATMEL_DEV_NAME, "Atmel_rtd_update_recvd_counter reached maximum value, resetting to 0");
        data->Atmel_rtd_update_recvd_counter = 0;
        }
        if (data->Atmel_rtd_update_recvd_counter % LOG_RTD_PROCESS_COUNT == 0)
        {
        mtouch_info (ATMEL_DEV_NAME, "Atmel_rtd_update_recvd_counter : %d", data->Atmel_rtd_update_recvd_counter);
        }
#endif
      proc_first_msg(data, msg);
      count++;
      break;

    case SECOND_MSG_PKT:
    case THIRD_MSG_PKT:
    case FOURTH_MSG_PKT:
      proc_sec_to_four_msg(data, msg);
      count++;
      break;

    case LAST_MSG_PKT:
      proc_last_msg(data, msg);
#ifdef BOSCH_RTC_2574261_HID_ENABLE_ATMEL_LOGGING_IMPROVEMENTS
        data->Atmel_rtd_update_prcsd_counter++;
        data->Atmel_rtd_intr_flag = 1;
        // Reset the counter if it reaches the maximum value to prevent overflow
        if (data->Atmel_rtd_update_prcsd_counter >= UINT32_MAX) {
            mtouch_warn(ATMEL_DEV_NAME, "rtd_update_prcsd_counter reached maximum value, resetting to 0");
        data->Atmel_rtd_update_prcsd_counter = 0;
        }
        if(data->Atmel_rtd_update_prcsd_counter % LOG_RTD_PROCESS_COUNT == 0)
        {
            mtouch_info(ATMEL_DEV_NAME, "Atmel RTD UPDATE PROCESSED COUNTER: %d", data->Atmel_rtd_update_prcsd_counter);
        }
#endif
      break;

    default:
      mtouch_error(ATMEL_DEV_NAME,"Invalid counter sequence");
  }
}

void proc_first_msg(struct mxt_data *data, uint8_t *msg)         //1st 9byte process
{
        if((data->T33_reportid_max - data->T33_reportid_min) != (RTD_REPORTID_COUNT -1))
        {
                mtouch_error(ATMEL_DEV_NAME,"Incorrect report_id T33_reportid_max=%X T33_reportid_min=%X",data->T33_reportid_max,data->T33_reportid_min);
                report_id_count_flag = true;
                return ;
        }

        if(msg[0] != data->T33_reportid_min)
        {
                mtouch_error(ATMEL_DEV_NAME,"Invalid report_id=%d for 1st messgae T33_reportid_min=%d",msg[0],data->T33_reportid_min);
                report_id_flag = true;
                return ;
        }

        memcpy(&data->touch_rtd[9*count], &msg[1], (T5_MSG_SIZE-1));
}

void proc_sec_to_four_msg(struct mxt_data *data, uint8_t *msg)         //next 9*3 byte process
{
        if(msg[0] != (data->T33_reportid_min+count))            //nonsequential report_id check
        {
                nonsequential_reportid_flag = true;
                mtouch_error(ATMEL_DEV_NAME,"Non-sequential report_id=%d",msg[0]);
                return ;
        }

        if(report_id_flag || report_id_count_flag)
        {
                return ;
        }

        memcpy(&data->touch_rtd[9*count], &msg[1], (T5_MSG_SIZE-1));
}

void proc_last_msg(struct mxt_data *data, uint8_t *msg)         //last 9byte process
{
    int ret = 0;
    if(msg[0] == data->T33_reportid_max)
    {
      if(!(report_id_flag || report_id_count_flag || nonsequential_reportid_flag))
      {
         memcpy(&data->touch_rtd[9*count], &msg[1], T5_LAST_MSG_SIZE);             //copying last 5bytes

             /* append group id to touch diag */
         if(data->display_group_id < 0)
         {
            ret = atmel_get_display_grpid(data);
            if(ret != 0)
            {
               mtouch_error(ATMEL_DEV_NAME, "failed to fetch group id: %d again", data->display_group_id);
            }
         }
         data->touch_rtd[MAX_RTD_DATA_LENGTH - 1] = data->display_group_id;
         data->touch_rtd_readflag = true;

         data->knob_rtd[0] = msg[7];                     //copying last 3-bytes for rtd_data
         data->knob_rtd[1] = msg[8];
         data->knob_rtd[2] = msg[9];

         data->knob_rtd_readflag = true;
       }
    }
        reset_state_variable();                                //clear all flags and variable used for check
}

void reset_state_variable()
{
        report_id_flag = false;
        report_id_count_flag = false;
        nonsequential_reportid_flag = false;
        count = 0;
}

static void mxt_proc_t9_message(struct mxt_data *data, uint8_t *message)
{
    int id;
    uint8_t status;
    int x;
    int y;
    int area;
    int amplitude;
    uint8_t vector;

    id = message[0] - data->T9_reportid_min;
    status = message[1];
    x = (message[2] << 4) | ((message[4] >> 4) & 0xf);
    y = (message[3] << 4) | ((message[4] & 0xf));

    /* Handle 10/12 bit switching */
    if (data->max_x < 1024)
        x >>= 2;
    if (data->max_y < 1024)
        y >>= 2;

    area = message[5];

    amplitude = message[6];
    vector = message[7];

    mtouch_debug(ATMEL_DEV_NAME,
        "[%u] %c%c%c%c%c%c%c%c x: %5u y: %5u area: %3u amp: %3u vector: %02X\n",
        id,
        (status & MXT_T9_DETECT) ? 'D' : '.',
        (status & MXT_T9_PRESS) ? 'P' : '.',
        (status & MXT_T9_RELEASE) ? 'R' : '.',
        (status & MXT_T9_MOVE) ? 'M' : '.',
        (status & MXT_T9_VECTOR) ? 'V' : '.',
        (status & MXT_T9_AMP) ? 'A' : '.',
        (status & MXT_T9_SUPPRESS) ? 'S' : '.',
        (status & MXT_T9_UNGRIP) ? 'U' : '.',
        x, y, area, amplitude, vector);

    /* prepare touch event message */
    data->touch_state[id].status = status;
    data->touch_state[id].x = x;
    data->touch_state[id].y = y;

    if (status & MXT_T9_DETECT) {
        /*
         * Multiple bits may be set if the host is slow to read
         * the status messages, indicating all the events that
         * have happened.
         */
        if (status & MXT_T9_RELEASE) {
            mtouch_driver_process_packet(data->inputevents_hdl, &data->touch_state[0], data, MTOUCH_PARSER_FLAG_NONE);
        }

        /* A size of zero indicates touch is from a linked T47 Stylus */
        if (area == 0)
            area = MXT_TOUCH_MAJOR_DEFAULT;

        /* if active, pressure must be non-zero */
        if (!amplitude)
            amplitude = MXT_PRESSURE_DEFAULT;

        /* Report Touch Event to mtouch */
        mtouch_driver_process_packet(data->inputevents_hdl, &data->touch_state[0], data, MTOUCH_PARSER_FLAG_NONE);
    } else {
        mtouch_driver_process_packet(data->inputevents_hdl, &data->touch_state[0], data, MTOUCH_PARSER_FLAG_NONE);
    }

    data->update_input = true;
}

int atmel_release_touch(struct mxt_data *dev)
{
    int Pending_release = 0;
    int count = 0;
    if (dev == NULL)
        return -1;

    /* Check and compare pending release events mapped to index */
    for(count=0; count < MXT_MAX_NUM_TOUCHPTS; count++)
    {
       if((dev->touch_prev_rec[count]) == 1)
       {
            Pending_release++;
            dev->touch_state[count].status = 0;
            dev->touch_prev_rec[count] = 0;
       }
    }
    /* If there is any pending release then process the packet */
    if(Pending_release > 0)
    {
        mtouch_error(ATMEL_DEV_NAME, "atmel_release_touch entered - Pending Release - %d\n",Pending_release);
        mtouch_driver_process_packet(dev->inputevents_hdl, &dev->touch_state[0], dev, MTOUCH_PARSER_FLAG_NONE);
    }

    return EOK;
}

static void mxt_proc_t100_message(struct mxt_data *data, uint8_t *message)
{
    int id;
    uint8_t status;
    uint8_t type = 0;
    uint16_t x;
    uint16_t y;
    int distance = 0;
    uint8_t major = 0;
    uint8_t minor = 0;
    uint8_t pressure = 0;
    uint8_t orientation = 0;
    uint8_t event_id;
    bool active = false;
    bool hover = false;
    const char *type_str = "UNKNOWN";
    id = message[0] - data->T100_reportid_min - 2;

    /* ignore SCRSTATUS events */
    if (id < 0 || id >= MXT_MAX_NUM_TOUCHPTS)
    {
        mtouch_error(ATMEL_DEV_NAME, "Invalid touch index %d\n", id);
        return;
    }

    status = message[1];

    event_id = status & 0x0F;
    if(event_id == 4)
    {
       data->touch_prev_rec[id] = 1;
    }
    //mtouch_error(ATMEL_DEV_NAME, "mxt_proc_t100_message id: %d, event: %x, data->touch_prev_rec: %u\n", id, event_id, data->touch_prev_rec[id]);

    x = UNALIGNED_RET16(&message[2]);
    y = UNALIGNED_RET16(&message[4]);

    if (status & MXT_T100_DETECT) {
        type = (status & MXT_T100_TYPE_MASK) >> 4;

        switch (type) {
        case MXT_T100_TYPE_HOVERING_FINGER:
            type_str = "MXT_T100_TYPE_HOVERING_FINGER";
            distance = MXT_DISTANCE_HOVERING;
            hover = true;
            active = true;
            break;

        case MXT_T100_TYPE_FINGER:
        case MXT_T100_TYPE_GLOVE:
            type_str = "MXT_T100_TYPE_FINGER";
            if (type == MXT_T100_TYPE_GLOVE)
            type_str = "MXT_T100_TYPE_GLOVE";
            distance = MXT_DISTANCE_ACTIVE_TOUCH;
            hover = false;
            active = true;

            if (data->t100_aux_area) {
                major = message[data->t100_aux_area + 1];
                minor = message[data->t100_aux_area];
            }

            if (data->t100_aux_ampl)
                pressure = message[data->t100_aux_ampl];

            if (data->t100_aux_vect)
                orientation = message[data->t100_aux_vect];

            break;

        case MXT_T100_TYPE_PASSIVE_STYLUS:
            type_str = "MXT_T100_TYPE_PASSIVE_STYLUS";
            distance = MXT_DISTANCE_ACTIVE_TOUCH;
            hover = false;
            active = true;

            /*
             * Passive stylus is reported with size zero so
             * hardcode.
             */
            major = MXT_TOUCH_MAJOR_DEFAULT;

            if (data->t100_aux_ampl)
                pressure = message[data->t100_aux_ampl];

            break;

        case MXT_T100_TYPE_ACTIVE_STYLUS:
            type_str = "MXT_T100_TYPE_ACTIVE_STYLUS";
            /* Report input buttons */
            //TODO report input buttons

            /* stylus in range, but position unavailable */
            if (!(message[6] & MXT_T107_STYLUS_HOVER))
                break;

            distance = MXT_DISTANCE_ACTIVE_TOUCH;
            active = true;
            major = MXT_TOUCH_MAJOR_DEFAULT;

            //TODO stylus suuport

            break;

        case MXT_T100_TYPE_LARGE_TOUCH:
            type_str = "MXT_T100_TYPE_LARGE_TOUCH";
            mtouch_debug(ATMEL_DEV_NAME, "Large Area Touch observed\n");
            break;

        default:
            type_str = "UNKNOWN";
            mtouch_debug(ATMEL_DEV_NAME, "Unexpected T100 type\n");
            return;
        }
    }
    /*
     * Values reported should be non-zero if tool is touching the
     * device
     */
    if (!pressure && !hover)
        pressure = MXT_PRESSURE_DEFAULT;

    /* prepare touch event message */
    data->touch_state[id].status = status;
    data->touch_state[id].x = x;
    data->touch_state[id].y = y;
    data->touch_state[id].pressure = pressure;
    data->touch_state[id].height = minor;
    data->touch_state[id].width = major;
    data->touch_state[id].orientation = orientation;

    if (active)
    {
        mtouch_debug(ATMEL_DEV_NAME, "[%u] type:%s x:%u y:%u w:%02X h:%02X p:%02X v:%02X\n",
        id, type_str, x, y, major, minor, pressure, orientation);
    }
    else
    {
        data->touch_prev_rec[id] = 0;
        mtouch_debug(ATMEL_DEV_NAME, "[%u] release x:%u y:%u\n", id, x, y);
    }

    //mtouch_debug(ATMEL_DEV_NAME, "[%u] touch_status active: %u, data->touch_prev_rec: %d", id, active, data->touch_prev_rec[id]);

    /* Report Touch Event to mtouch */
    mtouch_driver_process_packet(data->inputevents_hdl, &data->touch_state[0], data, MTOUCH_PARSER_FLAG_NONE);
    (void) distance;
}

static void mxt_proc_t6_messages(struct mxt_data *data, uint8_t *msg)
{
    uint8_t status = msg[1];
    uint32_t crc = msg[2] | (msg[3] << 8) | (msg[4] << 16);

    if (crc != data->config_crc) {
        data->config_crc = crc;
        mtouch_debug(ATMEL_DEV_NAME, "T6 Config Checksum: 0x%06X\n", crc);
    }

    /* Detect reset */
    if (status & MXT_T6_STATUS_RESET) {
        //Do proper action
    }

    /* Output debug if status has changed */
    if (status != data->t6_status)
        mtouch_debug(ATMEL_DEV_NAME, "T6 Status 0x%02X%s%s%s%s%s%s%s\n",
            status,
            status == 0 ? " OK" : "",
            status & MXT_T6_STATUS_RESET ? " RESET" : "",
            status & MXT_T6_STATUS_OFL ? " OFL" : "",
            status & MXT_T6_STATUS_SIGERR ? " SIGERR" : "",
            status & MXT_T6_STATUS_CAL ? " CAL" : "",
            status & MXT_T6_STATUS_CFGERR ? " CFGERR" : "",
            status & MXT_T6_STATUS_COMSERR ? " COMSERR" : "");

    /* Save current status */
    data->t6_status = status;
}

static void mxt_proc_t42_messages(struct mxt_data *data, uint8_t *msg)
{
    uint8_t status = msg[1];

    if (status & MXT_T42_MSG_TCHSUP)
        mtouch_info(ATMEL_DEV_NAME, "T42 suppress\n");
    else
        mtouch_info(ATMEL_DEV_NAME, "T42 normal\n");
}

static int mxt_proc_t48_messages(struct mxt_data *data, uint8_t *msg)
{
    uint8_t status, state;

    status = msg[1];
    state  = msg[4];

    mtouch_debug(ATMEL_DEV_NAME, "T48 state %d status %02X %s%s%s%s%s\n", state, status,
        status & 0x01 ? "FREQCHG " : "",
        status & 0x02 ? "APXCHG " : "",
        status & 0x04 ? "ALGOERR " : "",
        status & 0x10 ? "STATCHG " : "",
        status & 0x20 ? "NLVLCHG " : "");

    return 0;
}

static int mxt_proc_message(struct mxt_data *data, uint8_t *message)
{
    uint8_t report_id = message[0];
    bool dump = ((data->verbose > 7)? true: false);

    if (report_id == MXT_RPTID_RESERVED) {
        mtouch_error(ATMEL_DEV_NAME,
            "Received Reserved ReportID 0x00\n");
        error_memory("Atmel:Received Reserved ReportID 0x00\n");
        return -EINVAL;
    }
    if (report_id == MXT_RPTID_NOMSG) {
        mtouch_info(ATMEL_DEV_NAME,"No report ID msg\n");
        return 0;
    }

    if (report_id == data->T6_reportid) {
        mxt_proc_t6_messages(data, message);
    } else if (report_id >= data->T42_reportid_min
           && report_id <= data->T42_reportid_max) {
        mxt_proc_t42_messages(data, message);
    } else if (report_id == data->T48_reportid) {
        mxt_proc_t48_messages(data, message);
    } else if (!data->inputevents_hdl || data->suspended) {
        /*
         * Do not report events if input device is not
         * yet registered or returning from suspend
         */
        mxt_dump_message(data, message);
    } else if (report_id >= data->T9_reportid_min &&
           report_id <= data->T9_reportid_max) {
        mxt_proc_t9_message(data, message);
    } else if (report_id >= data->T100_reportid_min &&
           report_id <= data->T100_reportid_max) {
        mxt_dump_message(data, message);
        mxt_proc_t100_message(data, message);
    } else if (report_id == data->T19_reportid) {
        //TODO report T19 keys
    } else if (report_id == data->T25_reportid) {
        mxt_proc_t25_messages(data, message);
    } else if (report_id >= data->T15_reportid_min
           && report_id <= data->T15_reportid_max) {
        mxt_proc_t15_messages(data, message);
    } else if (report_id == data->T92_reportid) {
        mxt_proc_t92_messages(data, message);
    } else if (report_id == data->T93_reportid) {
        mxt_proc_t93_messages(data, message);
    } else if (report_id >= data->T152_reportid_min
           && report_id <= data->T152_reportid_max) {
        mxt_proc_t152_messages(data, message);
    } else if (report_id >= data->T33_reportid_min
           && report_id <= data->T33_reportid_max) {
        mxt_proc_t33_messages(data, message);
    } else {
        dump = true;
    }

    if (dump)
        mxt_dump_message(data, message);

    return 1;
}

static int mxt_read_and_process_messages(struct mxt_data *data, uint8_t count)
{
    int ret;
    int i;
    uint8_t num_valid = 0;

    /* Safety check for msg_buf */
    if (count > data->max_reportid)
        return -EINVAL;

    /* Process remaining messages if necessary */
    ret = data->i2c_funcs.read_reg(data->i2c_fd, data->T5_address,
                data->T5_msg_size * count, data->msg_buf);
    if (ret) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to read %u messages (%d)\n", count, ret);
        return ret;
    }

    for (i = 0;  i < count; i++) {
        ret = mxt_proc_message(data,
            data->msg_buf + data->T5_msg_size * i);

        if (ret < 0)
            return ret;
        if (ret == 1)
            num_valid++;
    }

    /* return number of messages read */
    return num_valid;
}

static uint8_t mxt_max_msg_read_count(struct mxt_data *data, uint8_t max_T5_msg_count)
{
    uint8_t T5_msg_count_limit = data->mtu / data->T5_msg_size;

    if (!data->mtu)
        return max_T5_msg_count;

    if (data->mtu < data->T5_msg_size) {
        mtouch_warn(ATMEL_DEV_NAME, "mtu set is lesser than the T5 message size\n");
        /* Return count of 1, as fallback */
        return 1;
    }
    /*
     * Return maximum number of T5 messages in single i2c transaction
     * based on "atmel,mtu" property.
     */
    return min(T5_msg_count_limit, max_T5_msg_count);
}

atmel_static int mxt_process_messages_t44(struct mxt_data *data)
{
    int ret;
    uint8_t T5_msg_count, total_pending;
    uint8_t total_processed = 0;
    int processed_valid = 0;

    /* Read T44 and T5 together */
    ret = data->i2c_funcs.read_reg(data->i2c_fd, data->T44_address,
                                   data->T5_msg_size + 1, data->msg_buf);
    if (ret < 0) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to read T44 and T5 \n");
        return ret;
    }

    T5_msg_count = data->msg_buf[0];

    /*
     * This condition may be caused by the CHG line being configured in
     * Mode 0. It results in unnecessary I2C operations but it is benign.
     */
    if (!T5_msg_count)
        return processed_valid;

    if (T5_msg_count > data->max_reportid) {
        mtouch_warn(ATMEL_DEV_NAME, "T44 count %d exceeded max report id\n",
             T5_msg_count);
        T5_msg_count = data->max_reportid;
    }
    /* Process first message */
    ret = mxt_proc_message(data, data->msg_buf + 1);
    if (ret < 0) {
        mtouch_warn(ATMEL_DEV_NAME, "Unexpected invalid message\n");
        return ret;
    }

    total_pending = T5_msg_count - 1;
    if (!total_pending) {
        processed_valid = 1;
        goto end;
    }

    /* Process remaining messages if necessary */
    T5_msg_count = mxt_max_msg_read_count(data, total_pending);

    do {
        if ((total_pending - total_processed) < T5_msg_count)
            T5_msg_count = total_pending - total_processed;
        ret = mxt_read_and_process_messages(data, T5_msg_count);
        if (ret < 0)
            goto end;
        /* Check for potential overflow before adding */
        if (processed_valid > (INT_MAX - ret)) {
            mtouch_error(ATMEL_DEV_NAME, "Integer overflow detected in processed_valid");
                    return -EINVAL;
        }
        total_processed += T5_msg_count;
        processed_valid += ret;

    } while (total_processed < total_pending);

    if (processed_valid != total_pending)
        mtouch_warn(ATMEL_DEV_NAME, "Unexpected invalid message\n");

end:
    return processed_valid;
}

static int mxt_process_messages_until_invalid(struct mxt_data *data)
{
    int count, read;
    int tries;

    count = mxt_max_msg_read_count(data, data->max_reportid);
    tries = (data->max_reportid / count) + 1;

    /* Read messages until we force an invalid */
    do {
        read = mxt_read_and_process_messages(data, count);
        if (read < 0)
            return read;
        else if (read < count)
            return 0;
        else ;
    } while (--tries);

    mtouch_error(ATMEL_DEV_NAME, "CHG pin isn't cleared\n");
    error_memory("Atmel_Touch: CHG pin isn't cleared\n");
    return -EBUSY;
}

static int mxt_process_messages(struct mxt_data *data)
{
    int total_handled, num_handled;
    uint8_t count = data->last_message_count;

    if (count < 1 || count > data->max_reportid)
        count = 1;

    /* include final invalid message */
    total_handled = mxt_read_and_process_messages(data, count + 1);
    if (total_handled < 0)
        return total_handled;
    /* if there were invalid messages, then we are done */
    else if (total_handled <= count)
        goto update_count;

    /* keep reading two msgs until one is invalid or reportid limit */
    do {
        num_handled = mxt_read_and_process_messages(data, 2);
        if (num_handled < 0)
            return num_handled;
        /* Check for potential overflow before adding */
        if (total_handled > (INT_MAX - num_handled)) {
           mtouch_error(ATMEL_DEV_NAME, "Integer overflow detected in total_handled");
                  return -EINVAL;
        }

        total_handled += num_handled;

        if (num_handled < 2)
            break;
    } while (total_handled < data->num_touchids);

update_count:
    data->last_message_count = total_handled;

    return total_handled;
}

static int mxt_set_t7_power_cfg(struct mxt_data *data, uint8_t sleep)
{
    int error;
    struct t7_config *new_config;
    struct t7_config deepsleep = { .active = 0, .idle = 0 };

    if (sleep == MXT_POWER_CFG_DEEPSLEEP)
        new_config = &deepsleep;
    else
        new_config = &data->t7_cfg;

    error = data->i2c_funcs.write_reg(data->i2c_fd, data->T7_address,
                sizeof(data->t7_cfg), new_config);
    if (error)
        return error;

    mtouch_debug(ATMEL_DEV_NAME, "Set T7 ACTV:%d IDLE:%d\n",
                 new_config->active, new_config->idle);

    return 0;
}

atmel_static int mxt_init_t7_power_cfg(struct mxt_data *data)
{
    int error;
    bool retry = false;

recheck:
    error = data->i2c_funcs.read_reg(data->i2c_fd, data->T7_address,
                sizeof(data->t7_cfg), &data->t7_cfg);
    if (error)
        return error;

    if (data->t7_cfg.active == 0 || data->t7_cfg.idle == 0) {
        if (!retry) {
            mtouch_debug(ATMEL_DEV_NAME, "T7 cfg zero, resetting\n");
            mxt_soft_reset(data);
            retry = true;
            goto recheck;
        } else {
            mtouch_debug(ATMEL_DEV_NAME, "T7 cfg zero after reset, overriding\n");
            data->t7_cfg.active = 20;
            data->t7_cfg.idle = 100;
            return mxt_set_t7_power_cfg(data, MXT_POWER_CFG_RUN);
        }
    }

    mtouch_debug(ATMEL_DEV_NAME, "Initialized power cfg: ACTV %d, IDLE %d\n",
        data->t7_cfg.active, data->t7_cfg.idle);
    return 0;
}

static int mxt_read_t9_resolution(struct mxt_data *data)
{
    int error;
    struct t9_range range;
    unsigned char orient;
    struct mxt_object *object;

    object = mxt_get_object(data, MXT_TOUCH_MULTI_T9);
    if (!object)
        return -EINVAL;

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T9_XSIZE,
                   sizeof(data->xsize), &data->xsize);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t9 x size:%d",error);
        return error;
    }

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T9_YSIZE,
                   sizeof(data->ysize), &data->ysize);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t9 y size:%d",error);
        return error;
    }

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T9_RANGE,
                   sizeof(range), &range);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t9 range:%d",error);
        return error;
    }

    data->max_x = ENDIAN_LE16(range.x);
    data->max_y = ENDIAN_LE16(range.y);
    data->max_x = UNALIGNED_RET16(&data->max_x);
    data->max_y = UNALIGNED_RET16(&data->max_y);

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                object->start_address + MXT_T9_ORIENT,
                1, &orient);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t9 orientation: %d",error);
        return error;
    }

    data->xy_switch = orient & MXT_T9_ORIENT_SWITCH;
    data->invertx = orient & MXT_T9_ORIENT_INVERTX;
    data->inverty = orient & MXT_T9_ORIENT_INVERTY;

    return 0;
}

static int mxt_read_t100_config(struct mxt_data *data)
{
    int error;
    struct mxt_object *object;
    uint16_t range_x, range_y;
    uint8_t cfg, tchaux;
    uint8_t aux;

    object = mxt_get_object(data, MXT_TOUCH_MULTITOUCHSCREEN_T100);
    if (!object)
        return -EINVAL;

    /* read touchscreen dimensions */
    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T100_XRANGE,
                   sizeof(range_x), &range_x);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t100 x range:%d",error);
        return error;
    }

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T100_YRANGE,
                   sizeof(range_y), &range_y);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t100 y range:%d",error);
        return error;
    }

    data->max_x = ENDIAN_LE16(range_x);
    data->max_y = ENDIAN_LE16(range_y);
    data->max_x = UNALIGNED_RET16(&data->max_x);
    data->max_y = UNALIGNED_RET16(&data->max_y);

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T100_XSIZE,
                   sizeof(data->xsize), &data->xsize);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t100 x size:%d",error);
        return error;
    }

    error = data->i2c_funcs.read_reg(data->i2c_fd,
                   object->start_address + MXT_T100_YSIZE,
                   sizeof(data->ysize), &data->ysize);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t100 y size:%d",error);
        return error;
    }

    /* read orientation config */
    error =  data->i2c_funcs.read_reg(data->i2c_fd,
                object->start_address + MXT_T100_CFG1,
                1, &cfg);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t100 orientation: %d",error);
        return error;
    }

    data->xy_switch = cfg & MXT_T100_CFG_SWITCHXY;
    data->invertx = cfg & MXT_T100_CFG_INVERTX;
    data->inverty = cfg & MXT_T100_CFG_INVERTY;

    /* allocate aux bytes */
    error =  data->i2c_funcs.read_reg(data->i2c_fd,
                object->start_address + MXT_T100_TCHAUX,
                1, &tchaux);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME,"failed to read t100 aux bytes: %d",error);
        return error;
    }

    aux = 6;

    if (tchaux & MXT_T100_TCHAUX_VECT)
        data->t100_aux_vect = aux++;

    if (tchaux & MXT_T100_TCHAUX_AMPL)
        data->t100_aux_ampl = aux++;

    if (tchaux & MXT_T100_TCHAUX_AREA)
        data->t100_aux_area = aux++;

    mtouch_debug(ATMEL_DEV_NAME,
        "T100 aux mappings vect:%u ampl:%u area:%u\n",
        data->t100_aux_vect, data->t100_aux_ampl, data->t100_aux_area);

    return 0;
}

atmel_static int mxt_initialize_multi_touch(struct mxt_data* data)
{
    int error;

    switch (data->multitouch) {
    case MXT_TOUCH_MULTI_T9:
        error = mxt_read_t9_resolution(data);
        if (error) {
            mtouch_error(ATMEL_DEV_NAME, "Failed to initialize T9 resolution\n");
            error_memory("Atmel_Touch: Failed to initialize T9 resolution\n");
            return error;
        }
        break;

    case MXT_TOUCH_MULTITOUCHSCREEN_T100:
        error = mxt_read_t100_config(data);
        if (error) {
            mtouch_error(ATMEL_DEV_NAME, "Failed to read T100 config\n");
            error_memory("Atmel_Touch: Failed to read T100 config\n");
            return error;
        }
        break;

    default:
        mtouch_error(ATMEL_DEV_NAME, "Invalid multitouch object\n");
        error_memory("Atmel_Touch: Invalid multitouch object\n");
        return -EINVAL;
    }

    /* Handle default values and orientation switch */
    if (data->max_x == 0)
        data->max_x = 1023;

    if (data->max_y == 0)
        data->max_y = 1023;

    if (data->xy_switch) {
        swap(data->max_x, data->max_y);
    }

    mtouch_info(ATMEL_DEV_NAME, "Touchscreen size X%uY%u\n", data->max_x, data->max_y);
    return 0;
}

atmel_static int mxt_check_retrigen(struct mxt_data *data)
{
    int error;
    int val;

    data->use_retrigen_workaround = false;

    /* //TODO if (irq_get_trigger_type(data->irq) & IRQF_TRIGGER_LOW)
        return 0; */

    if (data->T18_address) {
        error = data->i2c_funcs.read_reg(data->i2c_fd,
                       data->T18_address + MXT_COMMS_CTRL, 1, &val);
        if (error) {
            mtouch_error(ATMEL_DEV_NAME,"failed to use retrigen mode:%d",error);
            return error;
        }

        if (val & MXT_COMMS_RETRIGEN)
            return 0;
    }

    mtouch_warn(ATMEL_DEV_NAME, "Enabling RETRIGEN workaround\n");
    data->use_retrigen_workaround = true;
    return 0;
}

atmel_static int mxt_parse_object_table(struct mxt_data *data,
                  struct mxt_object *object_table)
{
    int i;
    uint8_t reportid;
    uint16_t end_address;

    /* Valid Report IDs start counting from 1 */
    reportid = 1;
    data->mem_size = 0;
    for (i = 0; i < data->info->object_num; i++) {
        struct mxt_object *object = object_table + i;
        uint8_t min_id, max_id;

        ENDIAN_LE16(object->start_address);

        if (object->num_report_ids) {
            min_id = reportid;
            reportid += object->num_report_ids *
                    mxt_obj_instances(object);
            max_id = reportid - 1;
        } else {
            min_id = 0;
            max_id = 0;
        }

        mtouch_debug(ATMEL_DEV_NAME,
            "T%u Start:%u Size:%zu Instances:%zu Report IDs:%u-%u\n",
            object->type, object->start_address,
            mxt_obj_size(object), mxt_obj_instances(object),
            min_id, max_id);

        switch (object->type) {
        case MXT_GEN_MESSAGE_T5:
            if (data->info->family_id == 0x80 &&
                data->info->version < 0x20) {
                /*
                 * On mXT224 firmware versions prior to V2.0
                 * read and discard unused CRC byte otherwise
                 * DMA reads are misaligned.
                 */
                data->T5_msg_size = mxt_obj_size(object);
            } else {
                /* CRC not enabled, so skip last byte */
                data->T5_msg_size = mxt_obj_size(object) - 1;
            }
            data->T5_address = object->start_address;
            break;
        case MXT_GEN_COMMAND_T6:
            data->T6_reportid = min_id;
            data->T6_address = object->start_address;
            break;
        case MXT_GEN_POWER_T7:
            data->T7_address = object->start_address;
            break;
        case MXT_SPT_DYNAMICCONFIGURATIONCONTAINER_T71:
            data->T71_address = object->start_address;
            break;
        case MXT_TOUCH_MULTI_T9:
            data->multitouch = MXT_TOUCH_MULTI_T9;
            /* Only handle messages from first T9 instance */
            data->T9_reportid_min = min_id;
            data->T9_reportid_max = min_id +
                        object->num_report_ids - 1;
            data->num_touchids = object->num_report_ids;
            break;
        case MXT_TOUCH_KEYARRAY_T15:
            data->T15_reportid_min = min_id;
            data->T15_reportid_max = max_id;
            break;
        case MXT_SPT_COMMSCONFIG_T18:
            data->T18_address = object->start_address;
            break;
        case MXT_SPT_SELFTEST_T25:
            data->T25_address = object->start_address;
            data->T25_reportid = min_id;
            break;
        case MXT_PROCI_TOUCHSUPPRESSION_T42:
            data->T42_reportid_min = min_id;
            data->T42_reportid_max = max_id;
            break;
        case MXT_SPT_MESSAGECOUNT_T44:
            data->T44_address = object->start_address;
            break;
        case MXT_SPT_GPIOPWM_T19:
            data->T19_reportid = min_id;
            break;
        case MXT_PROCG_NOISESUPPRESSION_T48:
            data->T48_reportid = min_id;
            break;
        case MXT_PROCI_SYMBOLGESTUREPROCESSOR:
            data->T92_reportid = min_id;
            data->T92_address = object->start_address;
            break;
        case MXT_PROCI_TOUCHSEQUENCELOGGER:
            data->T93_reportid = min_id;
            data->T93_address = object->start_address;
            break;
        case MXT_TOUCH_MULTITOUCHSCREEN_T100:
            data->multitouch = MXT_TOUCH_MULTITOUCHSCREEN_T100;
            data->T100_reportid_min = min_id;
            data->T100_reportid_max = max_id;
            /* first two report IDs reserved */
            if(object->num_report_ids >= 2) {
               data->num_touchids = object->num_report_ids - 2;
            } else {
                mtouch_error(ATMEL_DEV_NAME, "Invalid number of report IDs: %u", object->num_report_ids);
                data->num_touchids = 0; /* Set to 0 or handle appropriately */
            }
            break;
        case MXT_PROCI_ACTIVESTYLUS_T107:
            data->T107_address = object->start_address;
            break;
        case MXT_PROCI_KNOBDATA_T152:
            data->T152_reportid_min = min_id;
            data->T152_reportid_max = max_id;
            data->T152_address = object->start_address;
            break;
        case MXT_TOUCH_DIAGNOSTICS_T33:
            data->T33_reportid_min = min_id;
            data->T33_reportid_max = max_id;
            data->T33_address = object->start_address;
            break;
        }

        end_address = object->start_address
            + mxt_obj_size(object) * mxt_obj_instances(object) - 1;

        if (end_address >= data->mem_size)
            data->mem_size = end_address + 1;
    }

    /* Store maximum reportid */
    data->max_reportid = reportid;

    /* If T44 exists, T5 position has to be directly after */
    if (data->T44_address && (data->T5_address != data->T44_address + 1)) {
        mtouch_error(ATMEL_DEV_NAME, "Invalid T44 position\n");
        error_memory("Atmel_Touch: Invalid T44 position\n");
        return -EINVAL;
    }

    data->msg_buf = calloc(data->max_reportid, data->T5_msg_size);
    if (!data->msg_buf)
        return -ENOMEM;

    return 0;
}

static int mxt_read_info_block(struct mxt_data* data) {
    int error;
    size_t size;
    void *id_buf, *buf;
    uint8_t num_objects;
    uint32_t calculated_crc;
    uint8_t *crc_ptr;

    /* If info block already allocated, free it */
    if (data->info)
        mxt_free_object_table(data);

    /* Read 7-byte ID information block starting at address 0 */
    size = sizeof(struct mxt_info);
    id_buf = calloc(1, size);
    if (!id_buf)
        return -ENOMEM;

    error = data->i2c_funcs.read_reg(data->i2c_fd, MXT_INFO_BASE_REG, size, id_buf);
    if(error < 0)
        goto err_free_mem;

    /* Resize buffer to give space for rest of info block */
    num_objects = ((struct mxt_info *)id_buf)->object_num;
    size += (num_objects * sizeof(struct mxt_object)) + MXT_INFO_CHECKSUM_SIZE;

    buf = realloc(id_buf, size);
    if (!buf) {
        error = -ENOMEM;
        goto err_free_mem;
    }
    id_buf = buf;

    /* Read rest of info block */
    error = mxt_read_blks(data, MXT_OBJECT_START, size - MXT_OBJECT_START,
                          id_buf + MXT_OBJECT_START);
    if (error)
        goto err_free_mem;

    /* Extract & calculate checksum */
    crc_ptr = id_buf + size - MXT_INFO_CHECKSUM_SIZE;
    data->info_crc = crc_ptr[0] | (crc_ptr[1] << 8) | (crc_ptr[2] << 16);

    calculated_crc = mxt_calculate_crc(id_buf, 0, size - MXT_INFO_CHECKSUM_SIZE);

    /*
     * CRC mismatch can be caused by data corruption due to I2C comms
     * issue or else device is not using Object Based Protocol (eg i2c-hid)
     */
    if ((data->info_crc == 0) || (data->info_crc != calculated_crc)) {
        mtouch_error(ATMEL_DEV_NAME,
            "Info Block CRC error calculated=0x%06X read=0x%06X\n",
            calculated_crc, data->info_crc);
        error_memory( "Atmel_Touch: Info Block CRC error calculated=0x%06X read=0x%06X\n", calculated_crc, data->info_crc);
        error = -EIO;
        goto err_free_mem;
    }

    data->info = (struct mxt_info *)id_buf;

    mtouch_info(ATMEL_DEV_NAME,
         "Family: %u Variant: %u Firmware V%u.%u.%02X Objects: %u\n",
         data->info->family_id, data->info->variant_id,
         data->info->version >> 4, data->info->version & 0xf,
         data->info->build, data->info->object_num);

    /* Parse object table information */
    error = mxt_parse_object_table(data, id_buf + MXT_OBJECT_START);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME, "Error %d parsing object table\n", error);
        error_memory("Atmel_Touch: Error %d parsing object table\n", error);
        goto err_free_mem;
    }

    data->object_table = (struct mxt_object *)(id_buf + MXT_OBJECT_START);

    return 0;

err_free_mem:
    free(id_buf);
    return error;
}

#if 0
static int mxt_bootloader_write(struct mxt_data *data,
                const uint8_t * const val, unsigned int count)
{
    int ret;
//TODO bootloader read and writes
    struct i2c_msg msg;

    msg.addr = data->bootloader_addr;
    msg.flags = data->client->flags & I2C_M_TEN;
    msg.len = count;
    msg.buf = (uint8_t *)val;

    ret = i2c_transfer(data->client->adapter, &msg, 1);
    if (ret == 1) {
        ret = 0;
    } else {
        ret = ret < 0 ? ret : -EIO;
        dev_err(&data->client->dev, "%s: i2c send failed (%d)\n",
            __func__, ret);
    }
    ret = -1;
    return ret;
}
#endif

atmel_static int mxt_send_bootloader_cmd(struct mxt_data *data, bool unlock)
{
#if 0
    int ret;
    uint8_t buf[2];

    if (unlock) {
        buf[0] = MXT_UNLOCK_CMD_LSB;
        buf[1] = MXT_UNLOCK_CMD_MSB;
    } else {
        buf[0] = 0x01;
        buf[1] = 0x01;
    }

    ret = mxt_bootloader_write(data, buf, 2);
    if (ret)
        return ret;
#endif
    return -EIO;
}

static int mxt_lookup_bootloader_address(struct mxt_data *data, bool retry)
{
    uint8_t appmode = data->i2c_slave_addr;
    uint8_t bootloader;
    uint8_t family_id = data->info ? data->info->family_id : 0;

    switch (appmode) {
    case 0x4a:
    case 0x4b:
        /* Chips after 1664S use different scheme */
        if (retry || family_id >= 0xa2) {
            bootloader = appmode - 0x24;
            break;
        }
        /* Fall through for normal case */
    case 0x4c:
    case 0x4d:
    case 0x5a:
    case 0x5b:
        bootloader = appmode - 0x26;
        break;

    default:
        mtouch_error(ATMEL_DEV_NAME,
            "Appmode i2c address 0x%02x not found\n",
            appmode);
        error_memory(
            "Atmel_Touch: Appmode i2c address 0x%02x not found\n",
            appmode);
        return -EINVAL;
    }

    data->bootloader_addr = bootloader;
    return 0;
}

static int mxt_bootloader_read(struct mxt_data *data,
                   uint8_t *val, unsigned int count)
{
    int ret = -EIO;
#if 0
//TODO
    struct i2c_msg msg;

    msg.addr = data->bootloader_addr;
    msg.flags = data->client->flags & I2C_M_TEN;
    msg.flags |= I2C_M_RD;
    msg.len = count;
    msg.buf = val;

    rc = atmel->i2c_funcs.write(atmel->i2c_dev, msg.i2c_data.buflen, msg.i2c_data.buf);
    if (ret == 1) {
        ret = 0;
    } else {
        ret = ret < 0 ? ret : -EIO;
        dev_err(&data->client->dev, "%s: i2c recv failed (%d)\n",
            __func__, ret);
    }
#endif
    return ret;
}

static int mxt_probe_bootloader(struct mxt_data *data)
{
    int error;
    uint8_t buf[3];
    bool crc_failure, extended_id;

    /* Check bootloader status and version information */
    error = mxt_bootloader_read(data, buf, sizeof(buf));
    if (error)
        return error;

    crc_failure = (buf[0] & ~MXT_BOOT_STATUS_MASK) == MXT_APP_CRC_FAIL;
    extended_id = buf[0] & MXT_BOOT_EXTENDED_ID;

    mtouch_info(ATMEL_DEV_NAME, "Found bootloader addr:%02x ID:%u%s%u%s\n",
         data->bootloader_addr,
         extended_id ? (buf[1] & MXT_BOOT_ID_MASK) : buf[0],
         extended_id ? " version:" : "",
         extended_id ? buf[2] : 0,
         crc_failure ? ", APP_CRC_FAIL" : "");

    return 0;
}

static int mxt_bootloader_status(struct mxt_data *data)
{
    int error;

    error = mxt_lookup_bootloader_address(data, false);
    if (error) {
        mtouch_info(ATMEL_DEV_NAME,
             "Bootloader address is not specified\n");
        error_memory("Atmel_Touch: Bootloader address is not specified\n");
        return error;
    }
    /* Check bootloader state */
    error = mxt_probe_bootloader(data);
    if (error) {
        mtouch_info(ATMEL_DEV_NAME, "Trying alternate bootloader address\n");
        mxt_lookup_bootloader_address(data, true);
        error = mxt_probe_bootloader(data);
        if (error) {
            mtouch_error(ATMEL_DEV_NAME,
                "Chip is not in appmode or bootloader mode\n");
            error_memory("Atmel_Touch: Chip is not in appmode or bootloader mode\n");
            return error;
        }
    }
    return 0;
}

atmel_static int mxt_initialize(struct mxt_data* data)
{
    int error, recovery_attempts, rc;

    while (1) {
        error = mxt_read_info_block(data);
        if (!error)
            break;

        mtouch_info(ATMEL_DEV_NAME,
             "info block read failed (%d), so try bootloader method\n",
             error);
        error_memory("Atmel_Touch: info block read failed (%d), so try bootloader method\n",
             error);

        error = mxt_bootloader_status(data);
        if (error)
            return error;

        /* OK, we are in bootloader, see if we can recover */
        if (++recovery_attempts > 1) {
            mtouch_error(ATMEL_DEV_NAME, "Could not recover from bootloader mode\n");
            error_memory("Atmel_Touch: Could not recover from bootloader mode\n");
            /*
             * We can reflash from this state, so do not
             * abort initialization.
             */
            data->in_bootloader = true;
            return 0;
        }

        /* Attempt to exit bootloader into app mode */
        error = mxt_send_bootloader_cmd(data, false);
        if (error)
            return error;
        rc = safe_usleep(MXT_FW_RESET_TIME * 1000);
        if(rc != 0)
        {
            mtouch_error(ATMEL_DEV_NAME, "mxt_initialize:Timer error");
        }
    }

    error = mxt_check_retrigen(data);
    if (error)
        return error;

    error = mxt_init_t7_power_cfg(data);
    if (error) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to initialize power cfg\n");
        error_memory("Atmel_Touch: Failed to initialize power cfg\n");
        return error;
    }

    error = mxt_initialize_multi_touch(data);
    if (error)
        return error;

    return 0;
}

static int mxt_hw_init(struct mxt_data* atmel)
{
    uint8_t num_failures = 0;
    int rc = 0;
    int ret = 0;

    while (num_failures < atmel->max_retry_attempts) {
        rc = mxt_initialize(atmel);
        if ( rc == 0) {
            mtouch_info(ATMEL_DEV_NAME, "Atmel HW initialization successful");
            return EOK;
        }
        num_failures++;
        mtouch_error(ATMEL_DEV_NAME, "Atmel HW initialization failed,attmept-%d", num_failures);
        error_memory("Atmel_Touch: Atmel HW initialization failed,attmept-%d", num_failures);
        ret = safe_usleep(1 * 500 * 1000); /* 500 ms */
        if(ret != 0)
        {
            mtouch_error(ATMEL_DEV_NAME, "mxt_hw_init:Timer error");
        }
    }
    return rc;
}

atmel_static int mxt_start(struct mxt_data *data)
{
    int ret = 0;

    if (!data->suspended) {
        return 0;
    }
    if (data->in_bootloader) {
        return -EBUSY;
    }

    /*
     * Discard any touch messages still in message buffer
     * from before chip went to sleep
     */
    if (data->T44_address)
        ret = mxt_process_messages_t44(data);
    else
        ret = mxt_process_messages_until_invalid(data);

    if (ret < 0)
        goto end;

    /* set touch chip to run mode */
    ret = mxt_set_t7_power_cfg(data, MXT_POWER_CFG_RUN);
    if (ret)
        goto end;

    /* Recalibrate since chip has been in deep sleep */
    ret = mxt_t6_command(data, MXT_COMMAND_CALIBRATE, 1, false);
    if (ret)
        goto end;

    /* enable IRQ */
    InterruptUnmask(data->tp_intr, data->tp_iid);

end:
    if (!ret)
        data->suspended = false;

    return ret;
}

static int mxt_stop(struct mxt_data *data)
{
    int ret;

    if (data->suspended) {
        return 0;
    }
    if (data->in_bootloader) {
        return -EBUSY;
    }

    /* disable IRQ */
    InterruptMask(data->tp_intr, data->tp_iid);

    /* Set touch chip to deep sleep state */
    ret = mxt_set_t7_power_cfg(data, MXT_POWER_CFG_DEEPSLEEP);
    if (ret)
        return ret;

    data->suspended = true;

    return 0;
}

static int mxt_suspend(struct mxt_data *data)
{
    int rc = 0;

    rc = mxt_stop(data);
    if (rc) {
        mtouch_error(ATMEL_DEV_NAME, "mxt_stop failure %d", rc);
        error_memory("Atmel_Touch: mxt_stop failure %d", rc);
        return rc;
    }
  
    /* Release pending Touches */
    rc = atmel_release_touch(data);
    if(rc == -1)
       mtouch_error(ATMEL_DEV_NAME, "atmel_release_touch failure");

    /* stop device watchdog timers, {if any) */

    /* send notifications about suspend to other modules, (if any) */

    return 0;
}

static int mxt_resume(struct mxt_data *data)
{
    int rc = 0;
    uint8_t num_failures = 0;

    /* After suspend, FIDM MCU will turn off the power supply to touch IC.
     * It is recommended to re-initialize the touch controller after suspend */

    /* free the previously read object table for new initialization */
    mxt_free_object_table(data);

    /* perform hw initialization */

retry_mxt_hw_init_resume:

    rc = mxt_hw_init(data);

    if(rc != 0)
    {
        if(num_failures < MXT_MAX_RETRY_ATTEMPTS_ON_RESUME)
        {
            num_failures++;
            mtouch_error(ATMEL_DEV_NAME, "Atmel HW initialization on resume failed retrying");
            goto retry_mxt_hw_init_resume;
        }
    }

    if (rc) {
        mtouch_error(ATMEL_DEV_NAME, "Atmel HW initialization failed after resume %d", rc);
        error_memory("Atmel_Touch: Atmel HW initialization failed after resume %d", rc);
        return rc;
    }
    else{
        mtouch_info(ATMEL_DEV_NAME, "Atmel HW initialization on resume SUCCESS");
    }

    rc = mxt_start(data);
    if (rc) {
        mtouch_error(ATMEL_DEV_NAME, "mxt_start failure %d", rc);
        error_memory("Atmel_Touch: mxt_start failure %d", rc);
        return rc;
    }

    /* restart device watchdog timers, {if any) */

    /* send notifications about resume to other modules, (if any) */

    return 0;
}

void *atmel_ext_msg_handler(void *arg)
{
    struct mxt_data* atmel = arg;
    fidm_touch_msg_u msg;
    int rcvid;
    int ret;

    while (1) {
        rcvid = MsgReceive(atmel->msg_thread->fidm_attach->chid, &msg, sizeof(msg), NULL);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno)) {
                continue;
            }
            mtouch_error(ATMEL_DEV_NAME, "External thread MsgReceive failed: %s", strerror(errno));
            error_memory("Atmel_Touch: External thread MsgReceive failed: %s", strerror(errno));
            goto fail;
        } else if (rcvid != 0) {
            /* if atmel is in suspend/prepare state, return EAGAIN to try after sometime */
            if ((atmel->mxt_pm_state != PM_STATE_RESUME) && (msg.type != FIDM_TOUCH_GET_GROUPID)) {
                MsgError(rcvid, -EAGAIN);
                continue;
            }

            /* msg received */
            switch(msg.type) {
                case FIDM_TOUCH_GET_RTD_DATA:
                    if (atmel->verbose > 5)
                    {
                        mtouch_info(ATMEL_DEV_NAME, "Received FIDM_TD_MSG_GET_RTD_DATA");
                    }
#ifdef BOSCH_RTC_2574261_HID_ENABLE_ATMEL_LOGGING_IMPROVEMENTS
                    atmel->Atmel_fidm_RTD_rcvd_request_counter++;

                    // Check if the counter exceeds the maximum value and reset if necessary
                    if (atmel->Atmel_fidm_RTD_rcvd_request_counter == INT_MAX) {
                        atmel->Atmel_fidm_RTD_rcvd_request_counter = 0;
                        mtouch_info(ATMEL_DEV_NAME, "RTD received request counter reset to 0 after reaching INT_MAX");
                    }

                    if (atmel->touch_rtd_readflag == true) {
                        ret = MsgReply(rcvid, 0, atmel->touch_rtd, MAX_RTD_DATA_LENGTH);
                        if (ret == EOK) {
                        atmel->Atmel_fidm_RTD_prcsd_request_counter = (atmel->Atmel_fidm_RTD_prcsd_request_counter + 1) % INT_MAX;
                        atmel->touch_rtd_readflag = false;
                        }
#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
                        else if (errno == ESRCH) {
                            // Handle ESRCH error specifically
                            mtouch_error(ATMEL_DEV_NAME, "MsgReply failed: ESRCH - No such process");
                            MsgError(rcvid, -errno);
                        }
#endif
                        else {
                        MsgError(rcvid, -errno);
                        }
                        if (atmel->verbose > 5) {
                        mtouch_info(ATMEL_DEV_NAME, "Msg replied with rtd data: %d", errno);
                        }
                    }
                    else {
                        MsgError(rcvid, -EAGAIN);
                        atmel->Atmel_fidm_RTD_rcvd_request_error_counter = (atmel->Atmel_fidm_RTD_rcvd_request_error_counter + 1) % INT_MAX;
                    }
                    if (atmel->Atmel_fidm_RTD_rcvd_request_counter % LOG_RTD_REQUEST_COUNT == 0)
                    {
                        mtouch_info(ATMEL_DEV_NAME, "FIDM RTD received request counter: %d", atmel->Atmel_fidm_RTD_rcvd_request_counter);
                    }
                    if ((atmel->Atmel_fidm_RTD_prcsd_request_counter % LOG_RTD_REQUEST_COUNT == 0) &&
                        (atmel->Atmel_fidm_RTD_prcsd_bkup_counter != atmel->Atmel_fidm_RTD_prcsd_request_counter))
                    {
                        atmel->Atmel_fidm_RTD_prcsd_bkup_counter = atmel->Atmel_fidm_RTD_prcsd_request_counter;
                        if (atmel->touch_rtd_len >= 42)
                        { 
                        // Ensure array bounds are not exceeded
                        mtouch_info(ATMEL_DEV_NAME, "RTD= %x %x with length = %d",
                        atmel->touch_rtd[40], atmel->touch_rtd[41], atmel->touch_rtd_len);
                        } 
                        else 
                        {
                         mtouch_warn(ATMEL_DEV_NAME, "RTD data length insufficient for logging: %d", atmel->touch_rtd_len);
                        }
                        mtouch_info(ATMEL_DEV_NAME, "FIDM RTD processed request counter: %d, received error counter: %d",
                        atmel->Atmel_fidm_RTD_prcsd_request_counter, atmel->Atmel_fidm_RTD_rcvd_request_error_counter);
                    }
#else
                    if (atmel->touch_rtd_readflag == true) {
                         mtouch_info(ATMEL_DEV_NAME, "RTD= %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x with lenght = %d" ,\
                        atmel->touch_rtd[0],atmel->touch_rtd[1],atmel->touch_rtd[2],atmel->touch_rtd[3],atmel->touch_rtd[4],atmel->touch_rtd[5], atmel->touch_rtd[6],atmel->touch_rtd[7], atmel->touch_rtd[8],atmel->touch_rtd[9],\
                        atmel->touch_rtd[10],atmel->touch_rtd[11],atmel->touch_rtd[12],atmel->touch_rtd[13],atmel->touch_rtd[14],atmel->touch_rtd[15], atmel->touch_rtd[16],atmel->touch_rtd[17], atmel->touch_rtd[18],atmel->touch_rtd[19],\
                        atmel->touch_rtd[20],atmel->touch_rtd[21],atmel->touch_rtd[22],atmel->touch_rtd[23],atmel->touch_rtd[24],atmel->touch_rtd[25], atmel->touch_rtd[26],atmel->touch_rtd[27], atmel->touch_rtd[28],atmel->touch_rtd[29],\
                        atmel->touch_rtd[30],atmel->touch_rtd[31],atmel->touch_rtd[32],atmel->touch_rtd[33],atmel->touch_rtd[34],atmel->touch_rtd[35], atmel->touch_rtd[36],atmel->touch_rtd[37], atmel->touch_rtd[38],atmel->touch_rtd[39],\
                        atmel->touch_rtd[40],atmel->touch_rtd[41], atmel->touch_rtd_len);

                        ret = MsgReply(rcvid, 0, atmel->touch_rtd, MAX_RTD_DATA_LENGTH);
                        if (ret == EOK) {
                        atmel->touch_rtd_readflag = false;
                        }
#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
                        else if (errno == ESRCH) {
                            // Handle ESRCH error specifically
                            mtouch_error(ATMEL_DEV_NAME, "MsgReply failed: ESRCH - No such process");
                            MsgError(rcvid, -errno);
                        }
#endif
                        else {
                        MsgError(rcvid, -errno);
                        }
                        if (atmel->verbose > 5) {
                        mtouch_info(ATMEL_DEV_NAME, "Msg replied with rtd data: %d", errno);
                        }
                    }
                    else {
                        MsgError(rcvid, -EAGAIN);
                    }
#endif

                    break;
                case FIDM_TOUCH_GET_KNOB_RTD_DATA:
                    if (atmel->verbose > 5)
                        mtouch_info(ATMEL_DEV_NAME, "Received FIDM_TOUCH_GET_KNOB_RTD_DATA");
                    if (atmel->knob_rtd_readflag == true) {
                        ret = MsgReply(rcvid, 0, atmel->knob_rtd, atmel->knob_rtd_len);
                        if (ret == EOK) {
                            atmel->knob_rtd_readflag = false;
                        }
#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
                        else if (errno == ESRCH) {
                            // Handle ESRCH error specifically
                            mtouch_error(ATMEL_DEV_NAME, "MsgReply failed: ESRCH - No such process");
                            MsgError(rcvid, -errno);
                        }
#endif
                        else {
                            MsgError(rcvid, -errno);
                        }
                        if (atmel->verbose > 5)
                            mtouch_info(ATMEL_DEV_NAME, "Msg replied with knob rtd data: %d", errno);
                    }
                    else {
                        MsgError(rcvid, -EAGAIN);
                    }
                    break;
                 case FIDM_TOUCH_GET_GROUPID:
                     mtouch_info(ATMEL_DEV_NAME, "Received FIDM_TOUCH_GET_GROUPID");
                     if(atmel->display_group_id < 0) {
                         ret = atmel_get_display_grpid(atmel);
                         if(ret != 0)
                         {
                            mtouch_error(ATMEL_DEV_NAME, "failed to fetch group id, ret : %d", ret);
                            MsgError(rcvid, -errno);
                         } else {
                            ret = MsgReply(rcvid, 0, &atmel->display_group_id, sizeof(atmel->display_group_id));
                            if (ret != EOK) {
                                 MsgError(rcvid, -errno);
                            }
                         }
                     } else {
                         ret = MsgReply(rcvid, 0, &atmel->display_group_id, sizeof(atmel->display_group_id));
                         mtouch_info(ATMEL_DEV_NAME, "GROUPID %d ", atmel->display_group_id);
                         if(ret != EOK) {
                            MsgError(rcvid, -errno);
                         }
                     }
                     break;
                case FIDM_TOUCH_DISABLE_INTERRUPT:
                    if (atmel->verbose > 5)
                        mtouch_info(ATMEL_DEV_NAME, "Received FIDM_TOUCH_DISABLE_INTERRUPT");

                    InterruptMask(atmel->tp_intr, atmel->tp_iid);
                    MsgReply(rcvid, 0, NULL, 0);
                    break;

                case FIDM_TOUCH_ENABLE_INTERRUPT:
                    if (atmel->verbose > 5)
                        mtouch_info(ATMEL_DEV_NAME, "Received FIDM_TOUCH_ENABLE_INTERRUPT");

                    InterruptUnmask(atmel->tp_intr, atmel->tp_iid);
                    MsgReply(rcvid, 0, NULL, 0);
                    break;

                default:
                    if (atmel->verbose > 5)
                        mtouch_warn(ATMEL_DEV_NAME, "%s: Invalid msg id received", __FUNCTION__);
                    MsgError(rcvid, -ENOTSUP);
                    break;
            }
        }

    }

fail:
    mtouch_critical(ATMEL_DEV_NAME, "Failed to to configure external message handler");
    error_memory("Atmel_Touch: Failed to to configure external message handler");
    return NULL;

}

static void* tp_recv_thread(void* arg)
{
    struct mxt_data* atmel = arg;
    struct pm_ack_s ack = { .rc = 0 };
    mxt_ctrl_msg msg;
    int rcvid, ret;

    while (1) {
        rcvid = MsgReceive(atmel->attach->chid, &msg, sizeof(msg), NULL);
        if (rcvid < 0) {
            if ((EINTR == errno) || (ETIMEDOUT == errno)) {
                continue;
            }
            mtouch_error(ATMEL_DEV_NAME, "MsgReceive failed: %s", strerror(errno));
            error_memory("Atmel_Touch: MsgReceive failed: %s", strerror(errno));
            goto fail;
        } else if (rcvid == 0) {
            /* Pulse received */
            switch (msg.pulse.code) {
                case MXT_PULSE_CODE:
                if (atmel->verbose > 5) {
                    mtouch_info(ATMEL_DEV_NAME, "Received an interrupt (pulse) from the controller");
                }
#ifdef BOSCH_RTC_2574261_HID_ENABLE_ATMEL_LOGGING_IMPROVEMENTS
                atmel->Atmel_Tp_pulse_received_counter++;
                 // Reset the counter if it reaches the maximum value to prevent overflow
                if (atmel->Atmel_Tp_pulse_received_counter >= UINT32_MAX) {
                    mtouch_warn(ATMEL_DEV_NAME, "Tp_pulse_received_counter reached maximum value, resetting to 0");
                    atmel->Atmel_Tp_pulse_received_counter = 0;
                }
                /* Print the counter every 2 secs */
                if(atmel->Atmel_Tp_pulse_received_counter % LOG_RTD_REQUEST_COUNT == 0)
                {
                   mtouch_info (ATMEL_DEV_NAME, "Atmel_Tp_pulse_received_counter : %d", atmel->Atmel_Tp_pulse_received_counter);
                }
#endif
                if (!atmel->object_table) {
                    InterruptUnmask(atmel->tp_intr, atmel->tp_iid);
                    break;
                    }
                    if (atmel->T44_address)
                    ret = mxt_process_messages_t44(atmel);
                    else
                    ret = mxt_process_messages(atmel);
                    if (ret < 0) {
                    mtouch_error(ATMEL_DEV_NAME,"failed to process messages");
                    error_memory("Atmel_Touch: failed to process messages");
                    }
                    InterruptUnmask(atmel->tp_intr, atmel->tp_iid);

#ifdef BOSCH_RTC_2574261_HID_ENABLE_ATMEL_LOGGING_IMPROVEMENTS
                    atmel->Atmel_Tp_pulse_unmasked_counter++;
                // Reset the counter if it reaches the maximum value to prevent overflow
                if (atmel->Atmel_Tp_pulse_unmasked_counter >= UINT32_MAX) {
                    mtouch_warn(ATMEL_DEV_NAME, "Tp_pulse_unmasked_counter reached maximum value, resetting to 0");
                    atmel->Atmel_Tp_pulse_unmasked_counter = 0;
                }
                 if(atmel->Atmel_rtd_intr_flag == 1)
                {
                  atmel->Atmel_Tp_pulse_unmasked_rtd_counter++;
                   // Reset the counter if it reaches the maximum value to prevent overflow
                   if (atmel->Atmel_Tp_pulse_unmasked_rtd_counter >= UINT32_MAX) {
                    mtouch_warn(ATMEL_DEV_NAME, "Tp_pulse_unmasked_rtd_counter reached maximum value, resetting to 0");
                    atmel->Atmel_Tp_pulse_unmasked_rtd_counter = 0;
                   }
                   atmel->Atmel_rtd_intr_flag = 0;
                }
                if(atmel->Atmel_Tp_pulse_unmasked_counter % LOG_RTD_REQUEST_COUNT == 0)
                {
                   mtouch_info (ATMEL_DEV_NAME, "Tp_pulse_unmasked_counter : %d", atmel->Atmel_Tp_pulse_unmasked_counter);
                   mtouch_info (ATMEL_DEV_NAME, "Tp_pulse_unmasked_rtd_counter : %d", atmel->Atmel_Tp_pulse_unmasked_rtd_counter);
                }
#endif
                    break;

                case MXT_PM_PREPARE_PULSE:
                    mtouch_info(ATMEL_DEV_NAME, "Received MXT_PM_PREPARE_PULSE");
                    if(atmel->mxt_pm_state == PM_STATE_PREPARE)
                    {
                        mtouch_info(ATMEL_DEV_NAME, "Ignoring the MXT_PM_PREPARE_PULSE since mtouch driver is already in PREPARE state");

                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    }
                    else {

                        /* prepare mxt suspend */
                        // do nothing
                         if (ack.rc) {
                            mtouch_error(ATMEL_DEV_NAME, "Failed to handle MXT_PM_PREPARE_PULSE");
                            error_memory("Atmel_Touch: Failed to handle MXT_PM_PREPARE_PULSE");
                        } else {
                            /* send ACK to power manager */
                            atmel->mxt_pm_state = PM_STATE_PREPARE;
                        }

                    }
                    ack.state = PM_STATE_PREPARE;
                    ret = devctl(atmel->mxt_pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(ATMEL_DEV_NAME, "devctl(DCMD_PM_ACK) for MXT_PM_PREPARE_PULSE failed: %s\n", strerror(errno));
                        error_memory("Atmel_Touch: devctl(DCMD_PM_ACK) for MXT_PM_PREPARE_PULSE failed: %s\n", strerror(errno));
                    }
                    break;

                case MXT_PM_SUSPEND_PULSE:
                    mtouch_info(ATMEL_DEV_NAME, "Received MXT_PM_SUSPEND_PULSE");
                    if(atmel->mxt_pm_state == PM_STATE_SUSPEND)
                    {
                        mtouch_info(ATMEL_DEV_NAME, "Ignoring the MXT_PM_SUSPEND_PULSE since mtouch driver is already in SUSPEND state");

                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    }
                    else {
                        /* process mxt suspend */
                        ack.rc = mxt_suspend(atmel);
                        if (ack.rc) {
                            mtouch_error(ATMEL_DEV_NAME, "Failed to handle MXT_PM_SUSPEND_PULSE");
                            error_memory("Atmel_Touch: Failed to handle MXT_PM_SUSPEND_PULSE");
                        } else {
                            /* Update MXT PM state */
                            atmel->mxt_pm_state = PM_STATE_SUSPEND;
                        }
                    }
                    /* Send ACK to Power Manager */
                    ack.state = PM_STATE_SUSPEND;
                    ret = devctl(atmel->mxt_pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(ATMEL_DEV_NAME, "devctl(DCMD_PM_ACK) for MXT_PM_SUSPEND_PULSE failed: %s\n", strerror(errno));
                        error_memory("Atmel_Touch: devctl(DCMD_PM_ACK) for MXT_PM_SUSPEND_PULSE failed: %s\n", strerror(errno));
                    }
                    break;

                case MXT_PM_RESUME_PULSE:
                    mtouch_info(ATMEL_DEV_NAME, "Received MXT_PM_RESUME_PULSE");

                    /* Send ACK to Power Manager */
                    ack.state = PM_STATE_RESUME;
                    ret = devctl(atmel->mxt_pm_fd, DCMD_PM_ACK, &ack, sizeof(ack), NULL);
                    if (ret != EOK) {
                        mtouch_error(ATMEL_DEV_NAME, "devctl(DCMD_PM_ACK) for MXT_PM_RESUME_PULSE: failed: %s\n", strerror(errno));
                        error_memory("Atmel_Touch: devctl(DCMD_PM_ACK) for MXT_PM_RESUME_PULSE: failed: %s\n", strerror(errno));
                    }
                    if(atmel->mxt_pm_state == PM_STATE_RESUME)
                    {
                        mtouch_info(ATMEL_DEV_NAME, "Ignoring the MXT_PM_RESUME_PULSE since mtouch driver is already in RESUME state");

                        /* Send ACK to Power Manager */
                        ack.rc = 0;
                    }
                    else {
                        /* process mxt resume */
                        ack.rc = mxt_resume(atmel);
                        if (ack.rc) {
                            mtouch_error(ATMEL_DEV_NAME, "Failed to handle MXT_PM_RESUME_PULSE");
                            error_memory("Atmel_Touch: Failed to handle MXT_PM_RESUME_PULSE");
                        } else {
                            /* Update MXT PM state */
                            atmel->mxt_pm_state = PM_STATE_RESUME;
                        }
                    }
                    break;

                default:
                    mtouch_warn(ATMEL_DEV_NAME, "Received unknown pulse: %d", msg.pulse.code);
                    break;
            }

        }

    }

fail:
    mtouch_critical(ATMEL_DEV_NAME, "Failed to to configure TP after %d attempts, giving up", atmel->max_retry_attempts);
    error_memory("Atmel_Touch: Failed to to configure TP after %d attempts, giving up", atmel->max_retry_attempts);
    return NULL;
}

static int mxt_register_pm(struct mxt_data* data)
{
    int ret;
    struct pm_register_s pm_reg;

    data->mxt_pm_fd = open(data->pm_dev_name, O_RDWR  | O_CLOEXEC);
    if (data->mxt_pm_fd  == -1) {
        mtouch_error(ATMEL_DEV_NAME, "dev pm open() failed %d", strerror(errno));
        error_memory("Atmel_Touch: dev pm open() failed %s", strerror(errno));
        goto fail;
    }

    memset(&pm_reg, 0x0, sizeof(struct pm_register_s));
    INIT_PM_REGISTER_STRUCT(&pm_reg);

    strlcpy(pm_reg.name, ATMEL_DEV_NAME, sizeof(pm_reg.name));
    pm_reg.pulse_codes[PM_STATE_PREPARE] = MXT_PM_PREPARE_PULSE;
    pm_reg.pulse_codes[PM_STATE_SUSPEND] = MXT_PM_SUSPEND_PULSE;
    pm_reg.pulse_codes[PM_STATE_RESUME] = MXT_PM_RESUME_PULSE;
    pm_reg.pulse_codes[PM_STATE_COMPLETE] = -1;
    pm_reg.priority = PM_PRIO_LEVEL_0;
    pm_reg.flags = 0;
    pm_reg.chid = data->chid;

    /* Register Atmel with Power Manager */
    ret = devctl(data->mxt_pm_fd, DCMD_PM_REGISTER, &pm_reg, sizeof(struct pm_register_s), NULL);
    if (ret != EOK) {
        mtouch_error(ATMEL_DEV_NAME, "devctl() failed: %s\n", strerror(errno));
        error_memory("Atmel_Touch: devctl() failed: %s\n", strerror(errno));
        goto fail_pm_reg_err;
    }
    mtouch_info(ATMEL_DEV_NAME, "registering to %s is successful", data->pm_dev_name);

    /* Success */
    return 0;

fail_pm_reg_err:
    close(data->mxt_pm_fd);
fail:
    return -1;
}

static int set_option(const char* option, const char* value, void* arg)
{
    struct mxt_data* atmel = arg;
    if (0 == strcmp("width", option)) {
        return input_parse_unsigned(option, value, &atmel->width);
    } else if (0 == strcmp("height", option)) {
        return input_parse_unsigned(option, value, &atmel->height);
    } else if (0 == strcmp("invert_x", option)) {
        return input_parse_bool(option, value, &atmel->invert_x);
    } else if (0 == strcmp("invert_y", option)) {
        return input_parse_bool(option, value, &atmel->invert_y);
    } else if (0 == strcmp("swap_xy", option)) {
        return input_parse_bool(option, value, &atmel->swap_xy);
    } else if (0 == strcmp("i2c_dev_name", option)) {
        return input_parse_string(option, value, &atmel->i2c_devname);
    } else if (0 == strcmp("i2c_speed", option)) {
        return input_parse_unsigned(option, value, &atmel->i2c_speed);
    } else if (0 == strcmp("slave_addr", option)) {
        return input_parse_unsigned(option, value, &atmel->i2c_slave_addr);
    } else if (0 == strcmp("i2c_reg_addr_len", option)) {
        return input_parse_unsigned(option, value, &atmel->i2c_reg_addr_len);
    } else if (0 == strcmp("intr_gpio_pin", option)) {
        return input_parse_unsigned(option, value, &atmel->tp_intr_gpio);
    } else if (0 == strcmp("priority", option)) {
        return input_parse_unsigned(option, value, &atmel->thread_priority);
    } else if (0 == strcmp("config_file", option)) {
        return input_parse_string(option, value, &atmel->config_file);
    } else if (0 == strcmp("max_retry_attempts", option)) {
        return input_parse_unsigned(option, value, &atmel->max_retry_attempts);
    } else if (0 == strcmp("verbose", option)) {
        return input_parse_unsigned(option, value, &atmel->verbose);
    } else if (0 == strcmp("reset_gpio_pin", option)) {
        return input_parse_unsigned(option, value, &atmel->reset_gpio_pin);
    } else if (0 == strcmp("attach_point", option)) {
        return input_parse_string(option, value, &atmel->attach_point);
    } else if (0 == strcmp("fidm_attach_point", option)) {
        return input_parse_string(option, value, &atmel->msg_thread->fidm_attach_point);
    } else if (0 == strcmp("retry_delay", option)) {
        return input_parse_unsigned(option, value, &atmel->retry_delay);
    } else if (0 == strcmp("num_touchpts", option)) {
        return input_parse_unsigned(option, value, &atmel->num_touchids);
    } else if (0 == strcmp("mtu", option)) {
        return input_parse_unsigned(option, value, &atmel->mtu);
    } else if (0 == strcmp("pm_dev", option)) {
        return input_parse_string(option, value, &atmel->pm_dev_name);
    } else if (0 == strcmp("win_height", option)) {
        return input_parse_unsigned(option, value, &atmel->win_height);
    } else if (0 == strcmp("win_width", option)) {
        return input_parse_unsigned(option, value, &atmel->win_width);
    } else if (0 == strcmp("disp_id", option)) {
        return input_parse_unsigned(option, value, &atmel->disp_id);
    } else if (0 == strcmp("win_xpos", option)) {
        return input_parse_unsigned(option, value, &atmel->win_xpos);
    } else if (0 == strcmp("win_ypos", option)) {
        return input_parse_unsigned(option, value, &atmel->win_ypos);
    } else if (0 == strcmp("win_name", option)) {
        return input_parse_string(option, value, &atmel->win_name);
    } else if (0 == strcmp("dev_ctrl_path", option)) {
        return input_parse_string(option, value, &atmel->dev_ctrl_path);
    } else if (0 == strcmp("sync_point", option)) {
        return input_parse_unsigned(option, value, &atmel->sync_point);
    } else if (0 == strcmp("dev_status_path", option)) {
        return input_parse_string(option, value, &atmel->dev_status_path);
    }
    else {
        mtouch_error(ATMEL_DEV_NAME, "Invalid option: '%s'", option);
        return -1;
    }

    return -1;
}

void*
mtouch_driver_init(const char* options)
{
    pthread_attr_t pattr;
    sched_param_t param;
    int fd = -1;
    struct mxt_data* atmel = calloc(1, sizeof(*atmel));
    dispatch_t *dpp;
    int chid = -1;
    int error, ret;

    mtouch_info(ATMEL_DEV_NAME, "Initializing driver...");
    mtouch_info(ATMEL_DEV_NAME, "options: %s", options);

    if (InstallLocalErrMemReader(RINGBUFFER_SIZE) != 0)
    {
        mtouch_error(ATMEL_DEV_NAME, "errmem low priority worker thread creation failed !!!");
    }

    if (NULL == atmel) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to alloc device memory");
        error_memory("Atmel_Touch: Failed to alloc device memory");
        return NULL;
    }

    atmel->msg_thread = calloc (1, sizeof(*atmel->msg_thread));

    /* Initialize defaults */
    atmel->width = 1024;
    atmel->height = 1024;
    atmel->i2c_fd = -1;
    atmel->i2c_devname = NULL;
    atmel->i2c_speed = 400000;
    atmel->i2c_slave_addr = 0;
    atmel->i2c_reg_addr_len = 2;
    atmel->tp_intr = 0xFFFF;
    atmel->tp_intr_gpio = 0xFFFF;
    atmel->tp_iid = -1;
    atmel->thread_priority = 21;
    atmel->attach = NULL;
    atmel->attach_point = NULL;
    atmel->msg_thread->fidm_attach_point = NULL;
    atmel->coid = -1;
    atmel->faceplate_coid = -1;
    atmel->verbose = 0;
    atmel->num_touchids = MXT_MAX_NUM_TOUCHPTS;
    atmel->max_retry_attempts = MXT_MAX_RETRY_ATTEMPTS;
    atmel->mtu = 200;
    atmel->pm_dev_name = PM_DEV_NODE;
    /* diagnostics len */
    atmel->touch_rtd_len = MAX_RTD_DATA_LENGTH;
    atmel->knob_rtd_len = MAX_KNOB_RTD_DATA_LENGTH;
    atmel->touch_rtd_readflag = false;
    atmel->knob_rtd_readflag = false;
    atmel->display_group_id = -1;
    atmel->dev_status_path = "/dev/vcd/display-binder/status";

    /* window default to bosch display size*/
    atmel->win_height = 720;
    atmel->win_width = 1280;
    atmel->win_xpos = 0;
    atmel->win_ypos = 0;
    atmel->disp_id = 2;
    atmel->dev_ctrl_path = "/dev/vcd/display-binder/control";
    atmel->win_name = "qvm_window";
    atmel->sync_point = 1;
#ifdef BOSCH_RTC_2574261_HID_ENABLE_ATMEL_LOGGING_IMPROVEMENTS
    atmel->Atmel_fidm_RTD_rcvd_request_counter =0;
    atmel->Atmel_fidm_RTD_prcsd_request_counter =0;
    atmel->Atmel_fidm_RTD_rcvd_request_error_counter =0;
    atmel->Atmel_Tp_pulse_received_counter =0;
    atmel->Atmel_Tp_pulse_unmasked_counter =0;
    atmel->Atmel_rtd_update_recvd_counter =0;
    atmel->Atmel_rtd_update_prcsd_counter =0;
    atmel->Atmel_Tp_pulse_unmasked_rtd_counter =0;
    atmel->Atmel_rtd_intr_flag =0;
    atmel->Atmel_fidm_RTD_prcsd_bkup_counter =0;
#endif

    memset(atmel->touch_prev_rec, 0, sizeof(atmel->touch_prev_rec)); // Initialize touch_prev_rec to 0

    dpp = NULL;

    input_parseopts(options, set_option, atmel);

    /* Resource Manager Specific */
    if ((chid = ChannelCreate (_NTO_CHF_FIXED_PRIORITY)) == 0) {
        mtouch_error(ATMEL_DEV_NAME, "%s: ChannelCreate: %s", __FUNCTION__, strerror(errno));
        error_memory("Atmel_Touch: %s: ChannelCreate: %s", __FUNCTION__, strerror(errno));
        goto fail;
    }

    if ((dpp = dispatch_create_channel (chid, 0)) == NULL) {
        mtouch_error(ATMEL_DEV_NAME, "%s: dispatch_create_channel: %s", __FUNCTION__, strerror(errno));
        error_memory("Atmel_Touch: %s: dispatch_create_channel: %s", __FUNCTION__, strerror(errno));
        goto fail;
    }

    if (atmel->attach_point == NULL) {
        atmel->attach_point = strdup (ATTACH_POINT);
    }

    if (atmel->msg_thread->fidm_attach_point == NULL) {
        atmel->msg_thread->fidm_attach_point = strdup (FIDM_TOUCH_ATTACH_POINT);
    }

    /* Create channel for messages */
    if ((atmel->attach = name_attach(dpp, atmel->attach_point, 0)) == NULL) {
        mtouch_error(ATMEL_DEV_NAME, "%s: name_attach: %s", __FUNCTION__, strerror(errno));
        error_memory("Atmel_Touch: %s: name_attach: %s", __FUNCTION__, strerror(errno));
        goto fail;
    }

    atmel->chid = chid;
    atmel->coid = ConnectAttach(0, 0, atmel->attach->chid, _NTO_SIDE_CHANNEL, 0);
    if (-1 == atmel->coid) {
        mtouch_error(ATMEL_DEV_NAME, "%s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
        error_memory("Atmel_Touch: %s: ConnectAttach: %s", __FUNCTION__, strerror(errno));
        goto fail_connect;
    }

#ifdef BOSCH_RTC_2611049_HID_ENABLE_REPLY_BLOCKED_HANDLING
    /* Create channel for external messages */
    int fidm_chid = ChannelCreate(0); // Create a communication channel for external messages
    if (fidm_chid == -1) {
            mtouch_error(ATMEL_DEV_NAME, "%s", __FUNCTION__, strerror(errno));
            error_memory("Atmel_Touch: %s: ChannelCreate failed: %s", __FUNCTION__, strerror(errno));
            goto fail; // Exit if channel creation fails
    }
    dispatch_t* fidm_dpp = dispatch_create_channel(fidm_chid, 0); // Create a dispatch structure for the external channel
    if (fidm_dpp == NULL) {
            mtouch_error(ATMEL_DEV_NAME, "%s: dispatch_create_channel failed: %s", __FUNCTION__, strerror(errno));
            error_memory("Atmel_Touch: %s: dispatch_create_channel failed: %s", __FUNCTION__, strerror(errno));
            ChannelDestroy(fidm_chid); // Destroy the channel if dispatch creation fails
            goto fail;
    }
    if ((atmel->msg_thread->fidm_attach = name_attach(fidm_dpp, atmel->msg_thread->fidm_attach_point, 0)) == NULL) {
            mtouch_error(ATMEL_DEV_NAME, "%s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
            error_memory("Atmel_Touch: %s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
            //dispatch_destroy(fidm_dpp); // Dispatch destroy commented out, ensure cleanup elsewhere
            ChannelDestroy(fidm_chid); // Destroy the channel if name attachment fails
            goto fail;
    }
#else

    /* Create channel for external messages */
    if ((atmel->msg_thread->fidm_attach = name_attach(NULL, atmel->msg_thread->fidm_attach_point, 0)) == NULL) {
        mtouch_error(ATMEL_DEV_NAME, "%s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
        error_memory("Atmel_Touch: %s: Message handler thread name_attach: %s", __FUNCTION__, strerror(errno));
        goto fail_connect;
    }
#endif
    atmel->msg_thread->fidm_coid = ConnectAttach(0, 0, atmel->msg_thread->fidm_attach->chid, _NTO_SIDE_CHANNEL, 0);
    if (-1 == atmel->msg_thread->fidm_coid) {
        mtouch_error(ATMEL_DEV_NAME, "%s: Message handler thread ConnectAttach: %s", __FUNCTION__, strerror(errno));
        error_memory("Atmel_Touch: %s: Message handler thread ConnectAttach: %s", __FUNCTION__, strerror(errno));
        goto fail_fidm_connect;
    }

    /* get faceplatc connection id */
    ret = mxt_knob_connect_faceplate(atmel);
    if(ret != 0)
    {
        mtouch_error(ATMEL_DEV_NAME, "mtouch_driver_init: Connection failed to faceplate and shall retry during knob event, retval = %d\n", ret);
    }

    get_atmel_libi2c_funcs(&atmel->i2c_funcs);

    /* Attach to i2c */
    if (NULL != atmel->i2c_devname) {
        atmel->i2c_fd = atmel->i2c_funcs.open(atmel->i2c_devname, atmel->i2c_slave_addr, atmel->i2c_speed);
        if ((-1 == atmel->i2c_fd) || (atmel->i2c_fd < 0)) {
            mtouch_error(ATMEL_DEV_NAME,"failed to open i2c device: %d", error);
            error_memory("Atmel_Touch: failed to open i2c device: %s", strerror(errno));
            goto fail_i2c_connect;
        }
    } else {
        mtouch_error(ATMEL_DEV_NAME,"i2c_devname is not provided");
        error_memory("Atmel_Touch: i2c_devname is not provided");
        goto fail_i2c_connect;
    }

    /* Attach the TP interrupt */
    if (EOK != atmel_tp_intr_attach(atmel)) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to attach to interrupt");
        goto fail_interrupt_attach;
    }

    /* Register to /dev/pm for suspend/resume notifications */
    if (-1 == mxt_register_pm(atmel)) {
        mtouch_error(ATMEL_DEV_NAME, "%s: Failed to register with pm", __FUNCTION__);
    }

    if (EOK != mxt_hw_init(atmel)) {

        mtouch_error(ATMEL_DEV_NAME, "HW init failed putting driver in suspend state\n");
        atmel->mxt_pm_state = PM_STATE_SUSPEND;
        /* disable IRQ */
        InterruptMask(atmel->tp_intr, atmel->tp_iid);

        atmel->suspended = true;
    }
    else{
        atmel->mxt_pm_state = PM_STATE_RESUME;
        mtouch_info(ATMEL_DEV_NAME, "Atmel HW initialization SUCCESS");
    }

    /* Attach to libinputevents */
    if (EOK != attach_driver(atmel)) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to attach to libinputevents");
        goto fail_attach_mtouch;
    }

    if (mxt_create_qvm_window(atmel) < 0) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to create screen window");
        goto fail;
    }

    pthread_attr_init(&pattr);
    param.sched_priority = atmel->thread_priority;
    pthread_attr_setschedparam(&pattr, &param);
    pthread_attr_setinheritsched(&pattr, PTHREAD_EXPLICIT_SCHED);

    if (EOK != pthread_create(&atmel->recv_thread, &pattr, tp_recv_thread, atmel)) {
        mtouch_error(ATMEL_DEV_NAME, "Failed to create the intr thread");
        error_memory("Atmel_Touch: Failed to create the intr thread");
        goto fail_thread;
    }

    pthread_setname_np(atmel->recv_thread, ATMEL_DEV_NAME);

    if (EOK != pthread_create(&atmel->msg_thread->fidm_thread, NULL, atmel_ext_msg_handler, atmel)) {
       mtouch_error(ATMEL_DEV_NAME, "Failed to create the external message thread");
       goto fail_fidm_thread;
    }

    /* Display Group ID */
    ret = atmel_get_display_grpid(atmel);
    if(ret != 0)
    {
        mtouch_error(ATMEL_DEV_NAME, "Failed to fetch Display Group Id, try again %d", ret);
    }
    if (atmel->verbose > 5)
        atmel->debug_enabled = 1;

    /* Create file to signal that the driver is initialized and ready to serve clients */
    if (atmel->sync_point) {
        fd = open("/tmp/sync/mtouch_initialized", O_WRONLY | O_CREAT | O_TRUNC,
            S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH);
        if(fd == -1)
        {
            mtouch_error(ATMEL_DEV_NAME, "%s failed to create sync file", __FUNCTION__);
            error_memory("Atmel_Touch: %s failed to create sync file", __FUNCTION__);
        }
        else
        {
            close(fd);
        }
    } else {
        mtouch_info(ATMEL_DEV_NAME, "diagnostics sync point not enabled %d", atmel->sync_point);
    }
    return atmel;

fail_fidm_thread:
    pthread_cancel(atmel->recv_thread);
    pthread_join(atmel->recv_thread, NULL);

fail_thread:
    InterruptDetach(atmel->tp_iid);
    atmel->tp_iid = -1;

fail_interrupt_attach:
    mxt_free_object_table(atmel);

fail_attach_mtouch:
    atmel->i2c_funcs.close(atmel->i2c_fd);

fail_i2c_connect:
    ChannelDestroy(chid);
    chid = -1;
    ConnectDetach(atmel->coid);
    atmel->coid = -1;

fail_fidm_connect:
    name_detach(atmel->msg_thread->fidm_attach, 0);
    atmel->msg_thread->fidm_attach = NULL;

fail_connect:
    name_detach(atmel->attach, 0);
    atmel->attach = NULL;

fail:
    free(atmel);

    mtouch_error(ATMEL_DEV_NAME, "Driver initialization failure.");
    error_memory("Atmel_Touch: Driver initialization failure.");
    return NULL;
}

void
mtouch_driver_fini(void* dev)
{
    struct mxt_data* atmel = dev;

    pthread_cancel(atmel->recv_thread);
    pthread_join(atmel->recv_thread, NULL);
    pthread_cancel(atmel->msg_thread->fidm_thread);
    pthread_join(atmel->msg_thread->fidm_thread, NULL);

    if (atmel->tp_iid >= 0) {
        InterruptDetach(atmel->tp_iid);
        atmel->tp_iid = -1;
    }

    if (atmel->inputevents_hdl) {
        mtouch_driver_detach(atmel->inputevents_hdl);
        atmel->inputevents_hdl = NULL;
    }

    if (-1 != atmel->i2c_fd) {
        atmel->i2c_funcs.close(atmel->i2c_fd);
    }

    mxt_free_object_table(atmel);

    if (atmel->attach && atmel->attach->chid >= 0) {
        ChannelDestroy(atmel->chid);
        atmel->chid = -1;
        ConnectDetach(atmel->coid);
        atmel->coid = -1;
        name_detach(atmel->attach, 0);
        atmel->attach = NULL;
    }

    if (atmel->msg_thread->fidm_attach && atmel->msg_thread->fidm_attach->chid >= 0) {
        ChannelDestroy(atmel->msg_thread->fidm_attach->chid);
        atmel->msg_thread->fidm_attach->chid = -1;
        ConnectDetach(atmel->msg_thread->fidm_coid);
        atmel->msg_thread->fidm_coid = -1;
        name_detach(atmel->msg_thread->fidm_attach, 0);
        atmel->msg_thread->fidm_attach = NULL;
    }

    if (atmel->config_file) {
        free(atmel->config_file);
        atmel->config_file = NULL;
    }

    free(atmel);
}

#if defined(__QNXNTO__) && defined(__USESRCVERSION)
#include <sys/srcversion.h>
__SRCVERSION("$URL: http://svn.ott.qnx.com/product/branches/7.0.0/trunk/hardware/mtouch/atmel/atmel.c $ $Rev: 879934 $")
#endif
